#ifndef RIPEN_HPP
#define RIPEN_HPP

#include <functional>
#include <string>
#include <vector>
#include <unordered_map>
#include <exception>
#include <iostream>
#include <sstream>

namespace forth {

const char *version = "Ripen v4.1b (27 Feb 2024)";

using Num = double;
using Str = std::string;

typedef std::vector<Num> Stack;
typedef std::vector<Str> Code;

struct Context;

typedef std::function<void(Context&)> Word;
typedef std::function<void(Context&, const Str&)> Sigil;

struct Context {
	Stack stack;
	
	std::unordered_map<Str, Word> words;
	std::unordered_map<Str, Str> aliases;
	std::unordered_map<char, Sigil> sigils;
	std::unordered_map<Str, Num> variables;
	
	Str buf_to;
	Code buffer;
	Code code;
	Code loop;
	Str scratch;
	
	std::stringstream input;
	
	void execute(const Str& word, bool expand = true) {
		if (buf_to != "") {
			if (word != buf_to) {
				buffer.push_back(word);
			} else {
				words[word](*this);
				buf_to = "";
			}
		} else if (aliases.count(word) > 0 && expand) {
			execute(aliases[word], false);
		} else if (words.count(word) > 0) {
			words[word](*this);
		} else if (sigils.count(word[0]) > 0) {
			Str payload(word.substr(1));
			sigils.at(word[0])(*this, payload);
		} else {
			try {
				stack.push_back(std::stod(word));
			} catch (const std::invalid_argument& e) {
				Str msg("unknown word - ");
				throw std::domain_error(msg + word);
			}
		}
	}
	
	void interpret(const Code& code) {
		// Assume single-line comments were skipped by now.
		for (const auto& i: code)
			execute(i);
	}
	
	void interpret(const Str& input) {
		std::stringstream code(input);
		Str word;
		while(code >> word) {
			if (word == "\\")
				break;
			else
				execute(word);
		}
	}
};

struct UserWord {
	Code code;
	
	UserWord(const Code& init) { code = init; }
	
	void operator()(Context& ctx) const {
		for (const auto& i: code)
			ctx.execute(i); 
	}
};

inline Num pop(Stack& stack) {
	if (stack.size() > 0) {
		Num a = stack.back();
		stack.pop_back();
		return a;
	} else {
		throw std::out_of_range("stack underflow");
	}
}

inline Str join(const Code& data, char sep = ' ') {
	Str result;
	if (data.size() > 0) {
		for (const auto& i: data) {
			result += i;
			result += sep;
		}
		result.pop_back();
	}
	return result;
};

inline void load_builtins(Context& ctx) {
	ctx.words["depth"] = [](Context& c) {
		c.stack.push_back(c.stack.size());
	};
	ctx.words[".s"] = [](const Context& c) {
		for (const auto& i: c.stack)
			std::cout << i << " ";
		std::cout << std::endl;
	};
	ctx.words["words"] = [](Context& c) {
		for (const auto& i: c.words)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	ctx.words["sigils"] = [](Context& c) {
		for (const auto& i: c.sigils)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	ctx.words["aliases"] = [](Context& c) {
		for (const auto& i: c.aliases)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	
	ctx.words["/*"] = [](Context& c) { c.buf_to = "*/"; };
	ctx.words["*/"] = [](Context& c) { c.buffer.clear(); };

	ctx.words["["] = [](Context& c) {
		c.buffer.clear();
		c.buf_to = "]";
	};
	ctx.words["]"] = [](Context& c) { c.code = c.buffer; };

	ctx.words["{"] = [](Context& c) {
		c.buffer.clear();
		c.buf_to = "}";
	};
	ctx.words["}"] = [](Context& c) { c.code = c.buffer; };

	ctx.words["("] = [](Context& c) {
		c.buffer.clear();
		c.buf_to = ")";
	};
	ctx.words[")"] = [](Context& c) { c.scratch = join(c.buffer); };

	ctx.words["."] = [](Context& c) {
		std::cout << pop(c.stack) << " ";
	};
	ctx.words["emit"] = [](Context& c) {
		std::cout << (char) pop(c.stack);
	};
	ctx.words["space"] = [](Context& c) { std::cout << ' '; };
	ctx.words["spaces"] = [](Context& c) {
		const int n = pop(c.stack);
		for (int i = 0; i < n; i++) 
			std::cout << ' ';
	};
	ctx.words["cr"] = [](Context& c) {
		std::cout << std::endl;
	};
	ctx.words[".."] = [](Context& c) {
		std::cout << c.scratch << " ";
	};
	ctx.words["accept"] = [](Context& c) {
		Str line;
		getline(std::cin, line);
		c.input.str(line);
		c.input.seekg(0);
	};
	
	ctx.sigils['='] = [](Context& c, const Str& name) {
		c.variables[name] = pop(c.stack);
	};
	ctx.sigils['+'] = [](Context& c, const Str& name) {
		c.variables[name] += pop(c.stack);
	};
	ctx.sigils['-'] = [](Context& c, const Str& name) {
		c.variables[name] -= pop(c.stack);
	};
	ctx.sigils['$'] = [](Context& c, const Str& name) {
		c.stack.push_back(c.variables.at(name));
	};
	ctx.sigils['\''] = [](Context& c, const Str& name) {
		if (c.words.count(name) == 0) {
			Str msg("unknown word - ");
			throw std::domain_error(msg + name);
		} else {
			auto word = c.words.at(name).target<UserWord>();
			if (word) {
				c.code = word->code;
			} else {
				c.code.resize(1);
				c.code[0] = name;
			}
		}
	};
	ctx.sigils['/'] = [](Context& c, const Str& name) {
		c.words[name] = UserWord(c.code);
	};
	ctx.sigils['?'] = [](Context& c, const Str& name) {
		if (c.words.count(name) == 0) {
			Str msg("unknown word - ");
			throw std::domain_error(msg + name);
		} else {
			auto word = c.words.at(name).target<UserWord>();
			if (word)
				std::cout << join(word->code) << std::endl;
			else
				std::cout << "(native code)" << std::endl;
		}
	};
	
	ctx.words["interpret"] = [](Context& c) {
		c.interpret(c.input.str());
	};
	ctx.words["execute"] = [](Context& c) {
		c.interpret(c.code);
	};
	ctx.words["number"] = [](Context& c) {
		Num d;
		c.input >> d;
		c.stack.push_back(d);
	};
	ctx.words["parse-name"] = [](Context& c) {
		c.input >> c.scratch;
	};
	
	ctx.sigils[':'] = [](Context& c, const Str& name) {
		c.scratch = name;
		c.buffer.clear();
		c.buf_to = ";";
	};
	ctx.words[";"] = [](Context& c) {
		c.words[c.scratch] = UserWord(c.buffer);
	};
	ctx.sigils['@'] = [](Context& c, const Str& name) {
		c.aliases[name] = c.scratch;
	};
	
	ctx.words["+"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a + b);
	};
	ctx.words["-"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a - b);
	};
	ctx.words["*"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a * b);
	};
	ctx.words["/"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a / b);
	};
	ctx.words["mod"] = [](Context& c) {
		const int b = pop(c.stack);
		const int a = pop(c.stack);
		c.stack.push_back(a % b);
	};
	ctx.words["/mod"] = [](Context& c) {
		const int b = pop(c.stack);
		const int a = pop(c.stack);
		const auto result = std::div(a, b);
		c.stack.push_back(result.rem);
		c.stack.push_back(result.quot);
	};
	ctx.words["negate"] = [](Context& c) {
		c.stack.push_back(-pop(c.stack));
	};
	ctx.words["abs"] = [](Context& c) {
		const auto a = pop(c.stack);
		c.stack.push_back(a < 0 ? -a : a);
	};
	ctx.words["min"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a < b ? a : b);
	};
	ctx.words["max"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a > b ? a : b);
	};
	ctx.words["1+"] = [](Context& c) {
		c.stack.push_back(pop(c.stack) + 1);
	};
	ctx.words["1-"] = [](Context& c) {
		c.stack.push_back(pop(c.stack) - 1);
	};
	
	ctx.words["drop"] = [](Context& c) { pop(c.stack); };
	ctx.words["dup"] = [](Context& c) {
		if (c.stack.size() > 0)
			c.stack.push_back(c.stack.back());
		else
			throw std::out_of_range("Stack underflow in dup");
	};
	ctx.words["2dup"] = [](Context& c) {
		if (c.stack.size() > 1) {
			c.stack.push_back(c.stack[c.stack.size() - 2]);
			c.stack.push_back(c.stack[c.stack.size() - 2]);
		} else {
			throw std::out_of_range("Stack underflow in 2dup");
		}		
	};
	ctx.words["over"] = [](Context& c) {
		if (c.stack.size() > 1)
			c.stack.push_back(c.stack[c.stack.size() - 2]);
		else
			throw std::out_of_range("Stack underflow in over");	
	};
	ctx.words["swap"] = [](Context& c) {
		if (c.stack.size() > 1) {
			const auto size = c.stack.size();
			const auto temp = c.stack[size - 1];
			c.stack[size - 1] = c.stack[size - 2];
			c.stack[size - 2] = temp;
		} else {
			throw std::out_of_range("Stack underflow in swap");
		}		
	};
	ctx.words["rot"] = [](Context& c) {
		if (c.stack.size() > 2) {
			const auto temp = c.stack[c.stack.size() - 3];
			c.stack.erase(c.stack.end() - 3);
			c.stack.push_back(temp);
		} else {
			throw std::out_of_range("Stack underflow in rot");
		}		
	};
	ctx.words["-rot"] = [](Context& c) {
		if (c.stack.size() > 2) {
			const auto temp = pop(c.stack);
			c.stack.insert(c.stack.end() - 2, temp);
		} else {
			throw std::out_of_range("Stack underflow in -rot");
		}		
	};
	
	ctx.words["<"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a < b);
	};
	ctx.words["<="] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a <= b);
	};
	ctx.words[">"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a > b);
	};
	ctx.words[">="] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a >= b);
	};
	ctx.words["="] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a == b);
	};
	ctx.words["<>"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a != b);
	};
	
	ctx.words["and"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a && b);
	};
	ctx.words["or"] = [](Context& c) {
		const auto b = pop(c.stack);
		const auto a = pop(c.stack);
		c.stack.push_back(a || b);
	};
	ctx.words["not"] = [](Context& c) {
		c.stack.push_back(!pop(c.stack));
	};
	ctx.aliases["true"] = "1";
	ctx.aliases["false"] = "0";

	ctx.words["times"] = [](Context& c) {
		const Code body(c.code);
		const int limit = pop(c.stack);
		for (int i = 0; i < limit; i++) {
			c.variables["i"] = i;
			c.interpret(body);
		}
	};
	ctx.words["iftrue"] = [](Context& c) {
		const Code body(c.code);
		if (pop(c.stack))
			c.interpret(body);
	};
	ctx.words["iffalse"] = [](Context& c) {
		const Code body(c.code);
		if (!pop(c.stack))
			c.interpret(body);
	};

	ctx.words["while"] = [](Context& c) {
		c.loop = c.code;
		c.stack.push_back(false);
	};
	ctx.words["until"] = [](Context& c) {
		c.loop = c.code;
		c.stack.push_back(true);
	};
	ctx.words["repeat"] = [](Context& c) {
		const Code body(c.loop);
		const Code test(c.code);
		const auto limit = pop(c.stack);
		do {
			c.interpret(body);
			c.interpret(test);
		} while (pop(c.stack) != limit);
	};
}

}

#endif
