#ifndef RIPEN_HPP
#define RIPEN_HPP

#include <functional>
#include <string>
#include <vector>
#include <unordered_map>
#include <exception>
#include <iostream>
#include <sstream>

namespace forth {

const char *version = "Ripen v3.2 (14 Feb 2024)";

using Str = std::string;

typedef std::vector<double> Stack;
typedef std::vector<Str> Code;

struct Context;

typedef std::function<void(Context&)> Word;
typedef std::function<void(Context&, const Str&)> Sigil;

struct Context {
	Stack stack;
	
	std::unordered_map<Str, Word> words;
	std::unordered_map<Str, Str> aliases;
	std::unordered_map<char, Sigil> sigils;
	std::unordered_map<Str, double> variables;
	
	Str buf_to;
	Code buffer;
	Code code;
	Code loop;
	Str scratch;
	
	void execute(const Str& word, bool expand = true) {
		if (buf_to != "") {
			if (word != buf_to) {
				buffer.push_back(word);
			} else {
				words[word](*this);
				buf_to = "";
			}
		} else if (aliases.count(word) > 0 && expand) {
			execute(aliases[word], false);
		} else if (words.count(word) > 0) {
			words[word](*this);
		} else if (sigils.count(word[0]) > 0) {
			auto payload = word.substr(1);
			sigils.at(word[0])(*this, payload);
		} else {
			try {
				stack.push_back(std::stod(word));
			} catch (const std::invalid_argument& e) {
				Str msg = "unknown word - ";
				throw std::domain_error(msg + word);
			}
		}
	}
	
	void interpret(const Code& code) {
		for (const auto& i: code)
			execute(i);
	}
	
	void interpret(const Str& input) {
		std::stringstream code(input);
		Str word;
		while(code >> word) {
			if (word == "\\")
				break;
			else
				execute(word);
		}
	}
};

struct UserWord {
	Code code;
	
	UserWord(const Code& init) { code = init; }
	
	void operator()(Context& ctx) const {
		for (const auto& i: code)
			ctx.execute(i); 
	}
};

inline double pop(Stack& stack) {
	if (stack.size() > 0) {
		double a = stack.back();
		stack.pop_back();
		return a;
	} else {
		throw std::out_of_range("stack underflow");
	}
}

inline Str join(const Code& data, const Str& sep = " ") {
	Str result;
	if (data.size() > 0) {
		result += data.front();
		for (std::size_t i = 1; i < data.size(); i++) {
			result += sep;
			result += data[i];
		}
	}
	return result;
};

inline void load_builtins(Context& ctx) {
	ctx.words["depth"] = [](Context& ctx) {
		ctx.stack.push_back(ctx.stack.size());
	};
	ctx.words[".s"] = [](const Context& ctx) {
		for (const auto& i: ctx.stack)
			std::cout << i << " ";
		std::cout << std::endl;
	};
	ctx.words["words"] = [](Context& ctx) {
		for (const auto& i: ctx.words)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	ctx.words["sigils"] = [](Context& ctx) {
		for (const auto& i: ctx.sigils)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	ctx.words["aliases"] = [](Context& ctx) {
		for (const auto& i: ctx.aliases)
			std::cout << i.first << " ";
		std::cout << std::endl;
	};
	ctx.words["buffer"] = [](Context& ctx) {
		for (const auto& i: ctx.buffer)
			std::cout << i << " ";
		std::cout << std::endl;
	};
	
	ctx.words["/*"] = [](Context& ctx) { ctx.buf_to = "*/"; };
	ctx.words["*/"] = [](Context& ctx) { ctx.buffer.clear(); };

	ctx.words["["] = [](Context& ctx) {
		ctx.buffer.clear();
		ctx.buf_to = "]";
	};
	ctx.words["]"] = [](Context& ctx) { ctx.code = ctx.buffer; };

	ctx.words["{"] = [](Context& ctx) {
		ctx.buffer.clear();
		ctx.buf_to = "}";
	};
	ctx.words["}"] = [](Context& ctx) { ctx.code = ctx.buffer; };

	ctx.words["("] = [](Context& ctx) {
		ctx.buffer.clear();
		ctx.buf_to = ")";
	};
	ctx.words[")"] = [](Context& ctx) {
		ctx.scratch = join(ctx.buffer);
	};

	ctx.words["."] = [](Context& ctx) {
		std::cout << pop(ctx.stack) << " ";
	};
	ctx.words["space"] = [](Context& ctx) {
		std::cout << " ";
	};
	ctx.words["cr"] = [](Context& ctx) {
		std::cout << std::endl;
	};
	ctx.words[".."] = [](Context& ctx) {
		std::cout << ctx.scratch << " ";
	};
	ctx.words["accept"] = [](Context& ctx) {
		getline(std::cin, ctx.scratch);
	};
	
	ctx.sigils['='] = [](Context& ctx, const Str& name) {
		ctx.variables[name] = pop(ctx.stack);
	};
	ctx.sigils['+'] = [](Context& ctx, const Str& name) {
		ctx.variables[name] += pop(ctx.stack);
	};
	ctx.sigils['-'] = [](Context& ctx, const Str& name) {
		ctx.variables[name] -= pop(ctx.stack);
	};
	ctx.sigils['$'] = [](Context& ctx, const Str& name) {
		ctx.stack.push_back(ctx.variables.at(name));
	};
	ctx.sigils['\''] = [](Context& ctx, const Str& name) {
		ctx.scratch = name;
	};
	ctx.sigils['/'] = [](Context& ctx, const Str& name) {
		ctx.words[name] = UserWord(ctx.buffer);
	};
	ctx.sigils['?'] = [](Context& ctx, const Str& name) {
		if (ctx.words.count(name) == 0) {
			std::cerr << "unknown word - " << name << std::endl;
		} else {
			auto word = ctx.words.at(name).target<UserWord>();
			if (word)
				std::cout << join(word->code) << std::endl;
			else
				std::cout << "(native code)" << std::endl;
		}
	};
	
	ctx.words["interpret"] = [](Context& ctx) {
		ctx.interpret(ctx.scratch);
	};
	ctx.words["execute"] = [](Context& ctx) {
		ctx.words.at(ctx.scratch)(ctx);
	};
	ctx.words["number"] = [](Context& ctx) {
		ctx.stack.push_back(std::stod(ctx.scratch));
	};
	
	ctx.sigils[':'] = [](Context& ctx, const Str& name) {
		ctx.scratch = name;
		ctx.buffer.clear();
		ctx.buf_to = ";";
	};
	ctx.words[";"] = [](Context& ctx) {
		ctx.words[ctx.scratch] = UserWord(ctx.buffer);
	};
	ctx.sigils['@'] = [](Context& ctx, const Str& name) {
		ctx.aliases[name] = ctx.scratch;
	};
	
	ctx.words["+"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a + b);
	};
	ctx.words["-"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a - b);
	};
	ctx.words["*"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a * b);
	};
	ctx.words["/"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a / b);
	};
	ctx.words["mod"] = [](Context& ctx) {
		const int b = pop(ctx.stack);
		const int a = pop(ctx.stack);
		ctx.stack.push_back(a % b);
	};
	ctx.words["/mod"] = [](Context& ctx) {
		const int b = pop(ctx.stack);
		const int a = pop(ctx.stack);
		const auto result = std::div(a, b);
		ctx.stack.push_back(result.rem);
		ctx.stack.push_back(result.quot);
	};
	ctx.words["negate"] = [](Context& ctx) {
		ctx.stack.push_back(-pop(ctx.stack));
	};
	ctx.words["abs"] = [](Context& ctx) {
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a < 0 ? -a : a);
	};
	ctx.words["min"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a < b ? a : b);
	};
	ctx.words["max"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a > b ? a : b);
	};
	ctx.words["1+"] = [](Context& ctx) {
		ctx.stack.push_back(pop(ctx.stack) + 1);
	};
	ctx.words["1-"] = [](Context& ctx) {
		ctx.stack.push_back(pop(ctx.stack) - 1);
	};
	
	ctx.words["drop"] = [](Context& ctx) { pop(ctx.stack); };
	ctx.words["dup"] = [](Context& ctx) {
		if (ctx.stack.size() > 0)
			ctx.stack.push_back(ctx.stack.back());
		else
			throw std::out_of_range("Stack underflow in dup");
	};
	ctx.words["2dup"] = [](Context& ctx) {
		if (ctx.stack.size() > 1) {
			ctx.stack.push_back(ctx.stack[ctx.stack.size() - 2]);
			ctx.stack.push_back(ctx.stack[ctx.stack.size() - 2]);
		} else {
			throw std::out_of_range("Stack underflow in 2dup");
		}		
	};
	ctx.words["over"] = [](Context& ctx) {
		if (ctx.stack.size() > 1) {
			ctx.stack.push_back(ctx.stack[ctx.stack.size() - 2]);
		} else {
			throw std::out_of_range("Stack underflow in over");
		}		
	};
	ctx.words["swap"] = [](Context& ctx) {
		if (ctx.stack.size() > 1) {
			const auto size = ctx.stack.size();
			const auto temp = ctx.stack[size - 1];
			ctx.stack[size - 1] = ctx.stack[size - 2];
			ctx.stack[size - 2] = temp;
		} else {
			throw std::out_of_range("Stack underflow in swap");
		}		
	};
	ctx.words["rot"] = [](Context& ctx) {
		if (ctx.stack.size() > 2) {
			const auto temp = ctx.stack[ctx.stack.size() - 3];
			ctx.stack.erase(ctx.stack.end() - 3);
			ctx.stack.push_back(temp);
		} else {
			throw std::out_of_range("Stack underflow in rot");
		}		
	};
	ctx.words["-rot"] = [](Context& ctx) {
		if (ctx.stack.size() > 2) {
			const auto temp = pop(ctx.stack);
			ctx.stack.insert(ctx.stack.end() - 2, temp);
		} else {
			throw std::out_of_range("Stack underflow in -rot");
		}		
	};
	ctx.words["clear"] = [](Context& ctx) { ctx.stack.clear(); };
	
	ctx.words["<"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a < b);
	};
	ctx.words["<="] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a <= b);
	};
	ctx.words[">"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a > b);
	};
	ctx.words[">="] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a >= b);
	};
	ctx.words["="] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a == b);
	};
	ctx.words["<>"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a != b);
	};
	
	ctx.words["and"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a && b);
	};
	ctx.words["or"] = [](Context& ctx) {
		const auto b = pop(ctx.stack);
		const auto a = pop(ctx.stack);
		ctx.stack.push_back(a || b);
	};
	ctx.words["not"] = [](Context& ctx) {
		ctx.stack.push_back(!pop(ctx.stack));
	};
	ctx.aliases["true"] = "1";
	ctx.aliases["false"] = "0";

	ctx.words["times"] = [](Context& ctx) {
		const Code body(ctx.code);
		const int limit = pop(ctx.stack);
		for (int i = 0; i < limit; i++) {
			ctx.variables["i"] = i;
			ctx.interpret(body);
		}
	};
	ctx.words["iftrue"] = [](Context& ctx) {
		const Code body(ctx.code);
		if (pop(ctx.stack))
			ctx.interpret(body);
	};
	ctx.words["iffalse"] = [](Context& ctx) {
		const Code body(ctx.code);
		if (!pop(ctx.stack))
			ctx.interpret(body);
	};

	ctx.words["while"] = [](Context& ctx) {
		ctx.loop = ctx.code;
		ctx.stack.push_back(false);
	};
	ctx.words["until"] = [](Context& ctx) {
		ctx.loop = ctx.code;
		ctx.stack.push_back(true);
	};
	ctx.words["repeat"] = [](Context& ctx) {
		const Code body(ctx.loop);
		const Code test(ctx.code);
		const auto limit = pop(ctx.stack);
		do {
			ctx.interpret(body);
			ctx.interpret(test);
		} while (pop(ctx.stack) != limit);
	};
}

}

#endif
