#include <string>
#include <iostream>
#include <fstream>

#include "ripen.hpp"

const std::string help_text[] = {
	" [ -v | -h | -e CODE | -i FILE | PROGRAM ]",
	"",
	"Arguments:",
	"    PROGRAM:       run given program",
	"",
	"Options:",
	"    -h, --help:    show this message and quit",
	"    -v, --version: show version and quit",
	"    -e, --exec:    evaluate given CODE and quit",
	"    -i, --include: include given FILE and go interactive"
};

bool include(const std::string& filename, forth::Context& ctx) {
	std::ifstream source(filename);
	if (source.is_open()) {
		std::string line;
		while(std::getline(source, line))
			ctx.interpret(line);
		return true;
	} else {
		return false;
	}
}

bool try_to_include(const std::string& filename, forth::Context& ctx) {
	try {
		return include(filename, ctx);
	} catch (const std::domain_error& e) {
		std::cerr << "Domain error: " << e.what() << std::endl;
		return false;
	} catch (const std::bad_function_call& e) {
		std::cerr << "Bad function call: " << e.what() << std::endl;
		return false;
	} catch (const std::out_of_range& e) {
		std::cerr << "Out of range: " << e.what() << std::endl;
		return false;
	} catch (const std::runtime_error& e) {
		std::cerr << "Runtime error: " << e.what() << std::endl;
		return false;
	}
}

int main(int argc, char* argv[]) {
	forth::Context ctx;
	forth::load_builtins(ctx);

	ctx.words["included"] = [](forth::Context& ctx) {
		if (!include(ctx.scratch, ctx))
			throw std::runtime_error("can't open file");
	};
	ctx.words["bye"] = [](forth::Context& ctx) { std::exit(0); };

	std::vector<std::string> args;
	args.reserve(argc);
	for (int i = 0; i < argc; i++)
		args.push_back(argv[i]);

	bool interactive = false;

	if (args.size() < 2) {
		interactive = true;
	} else if (args[1] == "-v" || args[1] == "--version") {
		std::cout << forth::version << std::endl;
		return 0;
	} else if (args[1] == "-h" || args[1] == "--help") {
		std::cout << "Usage: " << args[0];
		for (const auto& i: help_text)
			std::cout << i << '\n';
		std::cout << std::flush;
		return 0;
	} else if (args[1] == "-e" || args[1] == "--exec") {
		if (args.size() > 2) {
			ctx.interpret(args[2]);
		} else {
			std::cerr << "Missing argument to "
				<< args[1] << std::endl;
			return 1;
		}
	} else if (args[1] == "-i" || args[1] == "--include") {
		if (args.size() < 3) {
			std::cerr << "Missing argument to "
				<< args[1] << std::endl;
			return 1;
		}
		if (try_to_include(args[2], ctx))
			std::cout << "Included " << args[2] << std::endl;
		else
			std::cerr << "Failed to include "
				<< args[2] << std::endl;
		interactive = true;
	} else if (!try_to_include(args[1], ctx)) {
		return 2;
	}
	
	if (!interactive) return 0;
	
	std::cout << "Welcome to " << forth::version << '\n';
	std::cout << "Words: " << ctx.words.size() << ' ';
	std::cout << "Sigils: " << ctx.sigils.size() << ' ';
	std::cout << "Aliases: " << ctx.aliases.size() << '\n';
	std::cout << "Enter code or 'bye' to exit:" << std::endl;
	std::cout << "? ";

	std::string line;
	
	while(std::getline(std::cin, line)) {
		try {
			ctx.interpret(line);
			std::cout << "ok\n";
		} catch (const std::domain_error& e) {
			std::cerr << "Domain error: "
				<< e.what() << std::endl;
		} catch (const std::bad_function_call& e) {
			std::cerr << "Bad function call: "
				<< e.what() << std::endl;
		} catch (const std::out_of_range& e) {
			std::cerr << "Out of range: "
				<< e.what() << std::endl;
		} catch (const std::runtime_error& e) {
			std::cerr << "Runtime error: "
				<< e.what() << std::endl;
		}
		std::cout << (ctx.buf_to == "" ? "? " : "... ");
	}
}
