## Ripen: stack-based scripting language focused on cleanliness.

import tables, strutils
export tables # Any importing module will need it to define new words.

type
        Prim* = proc() ## Primitive word implementation.
        Sigil* = proc(word: string) ## Sigil implementation.
        WordKind* = enum primWord, userWord

type Word* = object
        case kind*: WordKind
        of primWord: prim*: Prim
        of userWord: code*: seq[string]

var
        stack*: seq[float] ## The Ripen stack; made of floats because why not.
        words*: Table[string, Word] ## The main dictionary.
        alias*: Table[string, string] ## The alias dictionary.
        sigil*: Table[char, Sigil] ## The sigil dictionary.
        variables*: Table[string, float] ## The variable dictionary.
var
        buf_to*: string ## Set to non-nil to start buffering until given word.
        buffer*: seq[string] ## Words accumulate here while buffering.
        code*: seq[string] ## Lists of words are copied here after buffering.
        loop*: seq[string] ## Loop bodies are moved here to free the code.
        scratch*: string ## Scratchpad area for working with strings.

proc `$`*(w: Word): string =
        case w.kind
        of primWord: "(native code)"
        of userWord: w.code.join(" ")

proc interpret*(code_block: seq[string])

proc exec*(word: Word) =
        case word.kind
        of primWord: word.prim()
        of userWord: word.code.interpret()

proc interpret*(code_block: seq[string]) =
        ## The heart of Ripen.
        for w in code_block:
                if w == "\\":
                        break
                elif buf_to != "":
                        if w != buf_to:
                                buffer.add(w)
                        else:
                                words[w].exec
                                buf_to = ""
                elif w in alias:
                        let a = alias[w]
                        words[a].exec
                elif w in words:
                        words[w].exec
                elif w[0] in sigil:
                        sigil[w[0]](w[1..^1])
                else:
                        stack.add parseFloat(w)

template primitive*(p: Prim): Word = Word(kind: primWord, prim: p)

proc pstack() = echo stack.join(" ")
proc depth() = stack.add float(len(stack))
proc pwords() =
        for i in words.keys:
                write stdout, i, " "
proc palias() =
        for i in alias.keys:
                write stdout, i, " "
proc sigils() =
        for i in sigil.keys:
                write stdout, i, " "
proc pbuffer() = echo buffer.join(" ")

words[".s"] = primitive(pstack)
words["depth"] = primitive(depth)
words["words"] = primitive(pwords)
words["aliases"] = primitive(palias)
words["sigils"] = primitive(sigils)
words["buffer"] = primitive(pbuffer)

proc comment1() = buf_to = "*/"
proc comment2() = buffer.setLen 0
proc bracket1() = buffer.setLen 0; buf_to = "]"
proc bracket2() = code = buffer
proc brace1() = buffer.setLen 0; buf_to = "}"
proc brace2() = code = buffer
proc paren1() = buffer.setLen 0; buf_to = ")"
proc paren2() = scratch = buffer.join(" ")

words["/*"] = primitive(comment1)
words["*/"] = primitive(comment2)
words["["] = primitive(bracket1)
words["]"] = primitive(bracket2)
words["{"] = primitive(brace1)
words["}"] = primitive(brace2)
words["("] = primitive(paren1)
words[")"] = primitive(paren2)

sigil['='] = proc (n: string) = variables[n] = stack.pop
sigil['$'] = proc (n: string) = stack.add variables[n]
sigil['\''] = proc (n: string) = scratch = n
sigil['@'] = proc (n: string) = alias[n] = scratch
sigil['/'] = proc (n: string) =
        write stderr, "The / sigil is deprecated; use : to define words."
        words[n] = Word(kind: userWord, code: code)
sigil['?'] = proc (n: string) = echo words[n]

proc execute() = words[scratch].exec
proc number() = stack.add parseFloat(scratch)

words["execute"] = primitive(execute)
words["number"] = primitive(number)

sigil[':'] = proc (n: string) =
        scratch = n
        buffer.setLen 0
        buf_to = ";"
proc semicolon() = words[scratch] = Word(kind: userWord, code: buffer)
words[";"] = primitive(semicolon)

proc period() = write stdout, formatFloat(stack.pop, precision = -1), " "
proc pspace() = write stdout, " "
proc ptype() = write stdout, scratch, " "
proc cr() = write stdout, "\n"
proc accept() = scratch = stdin.readLine

words["."] = primitive(period)
words["space"] = primitive(pspace)
words[".."] = primitive(ptype)
words["cr"] = primitive(cr)
words["accept"] = primitive(accept)

proc add() = stack.add(stack.pop() + stack.pop())
proc sub() =
        let b = stack.pop
        let a = stack.pop
        stack.add(a - b)
proc mul() = stack.add(stack.pop() * stack.pop())
proc pdiv() =
        let b = stack.pop
        let a = stack.pop
        stack.add(a / b)
proc pmod() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(int(a) mod int(b))
proc slashmod() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(int(a) mod int(b))
        stack.add float(int(a) div int(b))
proc negate() = stack.add(-stack.pop)
proc pabs() = stack.add(abs(stack.pop))
proc pmin() = stack.add(min(stack.pop, stack.pop))
proc pmax() = stack.add(max(stack.pop, stack.pop))
proc one_plus() = stack.add(stack.pop() + 1)
proc one_minus() = stack.add(stack.pop() - 1)

words["+"] = primitive(add)
words["-"] = primitive(sub)
words["*"] = primitive(mul)
words["/"] = primitive(pdiv)
words["mod"] = primitive(pmod)
words["/mod"] = primitive(slashmod)
words["negate"] = primitive(negate)
words["abs"] = primitive(pabs)
words["min"] = primitive(pmin)
words["max"] = primitive(pmax)
words["1+"] = primitive(one_plus)
words["1-"] = primitive(one_minus)

proc drop() = discard stack.pop
proc dup() = stack.add stack[^1]
proc ddup() =
        stack.add stack[^2]
        stack.add stack[^2]
proc over() = stack.add(stack[^2])
proc pswap() = swap stack[^1], stack[^2]
proc rot() =
        let tmp = stack[^3]
        stack.delete(stack.len() - 3)
        stack.add tmp
proc rrot() =
        let tmp = stack.pop
        stack.insert(tmp, stack.len() - 2)
proc clear() = stack.setLen 0

words["drop"] = primitive(drop)
words["dup"] = primitive(dup)
words["2dup"] = primitive(ddup)
words["over"] = primitive(over)
words["swap"] = primitive(pswap)
words["rot"] = primitive(rot)
words["-rot"] = primitive(rrot)
words["clear"] = primitive(clear)

proc lt() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(a < b)
proc lte() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(a <= b)
proc gt() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(a > b)
proc gte() =
        let b = stack.pop
        let a = stack.pop
        stack.add float(a >= b)
proc eq() = stack.add float(stack.pop == stack.pop)
proc ne() = stack.add float(stack.pop != stack.pop)

words["<"] = primitive(lt)
words["<="] = primitive(lte)
words[">"] = primitive(gt)
words[">="] = primitive(gte)
words["="] = primitive(eq)
words["<>"] = primitive(ne)

proc pand() = stack.add(float(bool(stack.pop) and bool(stack.pop)))
proc por() = stack.add(float(bool(stack.pop) or bool(stack.pop)))
proc pnot() = stack.add float(not bool(stack.pop))
proc ptrue() = stack.add float(true)
proc pfalse() = stack.add float(false)

words["and"] = primitive(pand)
words["or"] = primitive(por)
words["not"] = primitive(pnot)
words["true"] = primitive(ptrue)
words["false"] = primitive(pfalse)

proc times() =
        let body = code
        let n = int(stack.pop)
        for i in 1 .. n:
                variables["i"] = float(i)
                interpret body
proc iftrue() =
        if bool(stack.pop): interpret code
proc iffalse() =
        if not bool(stack.pop): interpret code
proc pwhile() = loop = code; stack.add float(false)
proc puntil() = loop = code; stack.add float(true)
proc prepeat() =
        let body = loop
        let test = code
        let limit = stack.pop
        interpret body
        interpret test
        while stack.pop() != limit:
                interpret body
                interpret test

words["times"] = primitive(times)
words["iftrue"] = primitive(iftrue)
words["iffalse"] = primitive(iffalse)
words["while"] = primitive(pwhile)
words["until"] = primitive(puntil)
words["repeat"] = primitive(prepeat)

var atLine* = 0

proc included*(filename: string) =
        atLine = 1
        for i in filename.lines:
                interpret i.splitWhitespace
                inc atLine
proc pincluded() = scratch.included
proc pinterpret() = interpret scratch.splitWhitespace

words["included"] = primitive(pincluded)
words["interpret"] = primitive(pinterpret)

const version_string* = "Ripen Forth 3.0 beta (2023-07-19)"
const help_message* = """
Usage: ripen [options...] [files...]

Options:
        -h, --help:             show this help message and exit
        -v, --version:          show version string and exit
        -t, --test:             run built-in tests and exit
        -i, --interactive:      enter interactive mode after files
        -q, --quiet:            suppress banner, prompt, ok message
""" 

when isMainModule:
        from random import randomize, rand
        import parseopt, rdstdin
        
        var
                interactive = false
                quiet = false
                files: seq[string]

        proc psrand() = randomize int64(stack.pop)
        proc prand() = stack.add rand(1.0)
        proc bye() = quit 0
        
        words["srand"] = primitive(psrand)
        words["rand"] = primitive(prand)
        words["bye"] = primitive(bye)
                
        proc test() =
                let s = [
                        "3 2 .s + . cr words /* This is a comment */",
                        "3 =a $a . cr \\ This is a comment too",
                        ":hi ( Hello, world! ) .. cr ; ?hi hi",
                        "{ ( la ) .. } 3 times cr",
                        "3 2 + 3 2 * < { ( Yes! ) .. cr } iftrue"
                ]
                for i, j in s:
                        echo "Test #", i + 1, ": ", j 
                        interpret j.splitWhitespace
        
        let shorts = {'i', 'q', 'v', 'h', 't'}
        let longs = @["interactive", "quiet", "version", "help", "test"]
        var cmdline = initOptParser(shortNoVal = shorts, longNoVal = longs) 
        
        for kind, opt, val in cmdline.getopt:
                case kind
                of cmdArgument:
                        files.add opt
                of cmdShortOption, cmdLongOption:
                        case opt
                        of "h", "help":
                                echo help_message
                                quit 0
                        of "v", "version":
                                echo "Welcome to ", version_string
                                echo words.len, " words defined"
                                echo sigil.len, " sigils defined"
                                quit 0
                        of "t", "test":
                                test()
                                quit 0
                        of "i", "interactive":
                                interactive = true
                        of "q", "quiet":
                                quiet = true
                        else:
                                echo "Unknown option: ", opt
                                echo "Try ripen -h for instructions."
                                quit 1
                of cmdEnd:
                        stderr.write "Error reading the command line.\n"

        if files.len > 0:
                var inFile = ""
                try:
                        for i in files:
                                inFile = i
                                i.included
                except IOError:
                        stderr.write "Error including file: "
                        stderr.write getCurrentExceptionMsg(), "\n"
                        quit 2
                except IndexDefect:
                        stderr.write "Stack underflow"
                        stderr.write " in ", inFile, " line ", atLine, "\n"
                        quit 3
                except ValueError:
                        stderr.write getCurrentException().name, ": "
                        stderr.write getCurrentExceptionMsg(), "\n"
                        stderr.write " in ", inFile, " line ", atLine, "\n"
                if not interactive:
                        quit 0

        if not quiet:
                echo "Welcome to ", version_string
                echo words.len, " words and ", sigil.len, " sigils defined."
                echo "Type `bye` to leave the program."

        var line = ""
        var okay = true

        while okay:
                if quiet:
                        okay = readLineFromStdin("", line)
                else:
                        # TO DO: fix prompt to be dynamic.
                        okay = readLineFromStdin("> ", line)
                if not okay:
                        break
                try:
                        interpret line.splitWhitespace
                        if not quiet: echo "ok"
                except EOFError:
                        break
                except IndexDefect:
                        stderr.write "Stack underflow", "\n"
                except ValueError, IOError:
                        stderr.write getCurrentException().name, ": "
                        stderr.write getCurrentExceptionMsg(), "\n"
