#!/usr/bin/env python3

stack = []
words = {}
sigil = {}

buffer = []
buf_to = None

def interpret(block):
	global buf_to
	for w in block:
		if buf_to != None:
			if w != buf_to:
				buffer.append(w)
			else:
				words[w]()
				buf_to = None
		elif w in words:
			words[w]()
		elif w[0] in sigil:
			sigil[w[0]](w[1:])
		else:
			stack.append(float(w))

class UserWord:
	def __init__(self, code):
		self.code = code
	def __call__(self):
		interpret(self.code)
	def __str__(self):
		return "User word: " + " ".join(self.code)

words[".s"] = lambda: print(" ".join([str(i) for i in stack]))
words["depth"] = lambda: stack.append(len(stack))
words["words"] = lambda: print(" ".join(words.keys()))
words["sigils"] = lambda: print(" ".join(sigil.keys()))
words["buffer"] = lambda: print(" ".join(buffer))

def slash_star():
	global buf_to
	buf_to = "*/"
def star_slash():
	buffer.clear()
words["/*"] = slash_star
words["*/"] = star_slash

def open_bracket():
	global buf_to
	buffer.clear()
	buf_to = "]"
def close_bracket():
	stack.append(buffer.copy())
words["["] = open_bracket
words["]"] = close_bracket

def open_brace():
	global buf_to
	buffer.clear()
	buf_to = "}"
def close_brace():
	stack.append(buffer.copy())
words["{"] = open_brace
words["}"] = close_brace

def open_paren():
	global buf_to
	buffer.clear()
	buf_to = ")"
def close_paren():
	stack.append(" ".join(buffer))
words["("] = open_paren
words[")"] = close_paren

sigil[">"] = lambda n: words.__setitem__(n, stack.pop())
sigil["$"] = lambda n: stack.append(words[n])
sigil["'"] = lambda n: stack.append(n)
sigil["/"] = lambda n: words.__setitem__(n, UserWord(stack.pop()))

words["execute"] = lambda: stack.pop()()

def colon(n):
	global buf_to
	stack.append(n)
	buffer.clear()
	buf_to = ";"
sigil[":"] = colon
words[";"] = lambda: words.__setitem__(
	stack.pop(), UserWord(buffer.copy()))

words["."] = lambda: print(stack.pop(), end="")
words["space"] = lambda: print(" ", end="")
words["cr"] = lambda: print()

def subtract():
	b = stack.pop()
	a = stack.pop()
	stack.append(a - b)
def divide():
	b = stack.pop()
	a = stack.pop()
	stack.append(a / b)
def modulo():
	b = stack.pop()
	a = stack.pop()
	stack.append(a % b)
def divmod():
	b = stack.pop()
	a = stack.pop()
	stack.append(a % b)
	stack.append(a // b)
def incr():
	stack[-1] += 1
def decr():
	stack[-1] -= 1
words["+"] = lambda: stack.append(stack.pop() + stack.pop())
words["-"] = subtract
words["*"] = lambda: stack.append(stack.pop() * stack.pop())
words["/"] = divide
words["mod"] = modulo
words["/mod"] = divmod
words["negate"] = lambda: stack.append(-stack.pop())
words["abs"] = lambda: stack.append(abs(stack.pop()))
words["min"] = lambda: stack.append(min(stack.pop(), stack.pop()))
words["max"] = lambda: stack.append(max(stack.pop(), stack.pop()))
words["1+"] = incr
words["1-"] = decr

def ddup():
	stack.append(stack[-2])
	stack.append(stack[-2])
def swap():
	stack[-1], stack[-2] = stack[-2], stack[-1]
words["drop"] = lambda: stack.pop()
words["dup"] = lambda: stack.append(stack[-1])
words["2dup"] = ddup
words["over"] = lambda: stack.append(stack[-2])
words["swap"] = swap
words["rot"] = lambda: stack.append(stack.pop(-3))
words["-rot"] = lambda: stack.insert(-2, stack.pop())
words["clear"] = lambda: stack.clear()

def lt():
	b = stack.pop()
	a = stack.pop()
	stack.append(a < b)
def lte():
	b = stack.pop()
	a = stack.pop()
	stack.append(a <= b)
def gt():
	b = stack.pop()
	a = stack.pop()
	stack.append(a > b)
def gte():
	b = stack.pop()
	a = stack.pop()
	stack.append(a >= b)
words["<"] = lt
words["<="] = lte
words[">"] = gt
words[">="] = gte
words["="] = lambda: stack.append(stack.pop() == stack.pop())
words["<>"] = lambda: stack.append(stack.pop() != stack.pop())

words["and"] = lambda: stack.append(stack.pop() and stack.pop())
words["or"] = lambda: stack.append(stack.pop() or stack.pop())
words["not"] = lambda: stack.append(not stack.pop())
words["true"] = lambda: stack.append(True)
words["false"] = lambda: stack.append(False)

def times():
	n = int(stack.pop())
	code = stack.pop()
	for i in range(n):
		words["i"] = i
		interpret(code)
def iftrue():
	code = stack.pop()
	if stack.pop():
		interpret(code)
def iffalse():
	code = stack.pop()
	if not stack.pop():
		interpret(code)
words["times"] = times
words["iftrue"] = iftrue
words["iffalse"] = iffalse

def repeat():
	test = stack.pop()
	limit = stack.pop()
	body = stack.pop()
	interpret(body)
	interpret(test)
	while stack.pop() != limit:
		interpret(body)
		interpret(test)
words["repeat"] = repeat
words["while"] = lambda: stack.append(False)
words["until"] = lambda: stack.append(True)

def included():
	with open(stack.pop(), "r") as f:
		for i in f:
			interpret(i.split())
words["included"] = included

if __name__ == "__main__":
	import sys

	if len(sys.argv) > 1:
		try:
			interpret(sys.argv[1:])
			print("ok")
			sys.exit(0)
		except ValueError as e:
			print("Unknown word:", e, file=sys.stderr)
			sys.exit(1)
		except IndexError as e:
			print("Stack underflow:", e, file=sys.stderr)
			sys.exit(2)
	
	try:
		import readline
	except ImportError as e:
		print("(Command line editing is unavailable.)\n")

	words["bye"] = lambda: sys.exit()

	print("Welcome to Ripen v2019-11-28")
	print(len(words), "words and", len(sigil), "sigils defined")

	while True:
		try:
			interpret(input("? ").split())
			print("ok")
		except ValueError as e:
			print("Unknown word:", e, file=sys.stderr)
		except IndexError as e:
			print("Stack underflow:", e, file=sys.stderr)
		except EOFError as e:
			break
