unit forth; {$mode objFPC} {$H+}

{< Toy scripting language with a stack, postfix notation and Logo-like control structures.}

interface

uses
	strutils, sysutils, classes, contnrs;
const
	whitespace = [#9, #10, #11, #12, #13, ' '];
	version_string = '1.3.1 beta';
type
	PForthWord = ^TForthWord;
	TExecToken = procedure(const w: PForthWord);
	TForthWord = record
		exec: TExecToken;
		proc: TProcedure;
		code: TStringList;
		data: real;
	end;
	EForthError = class(Exception); {< Mainly for stack / syntax errors. }
var
	dictionary: TFPDataHashTable;
	stack: array of real;
	word_stack: array of PForthWord; {< Used for variables and such. }
	stack_pointer, word_pointer: integer; {< Right after the last element. }
	input_buffer, parse_buffer: TStringList; {< Always parse word by word. }
	code_buffer, loop_buffer: TStringList; {< Used in control structures. }
	parse_guard: string; {< Word where buffering state ends. }
	latest: string; {< Name of the most recently defined word. }
	state: (interpreting, buffering);

procedure def_builtin(const name: string; const proc: TProcedure);
procedure def_const(const name: string; const data: real);
procedure def_value(const name: string; const data: real);
procedure def_var(const name: string);
procedure def_colon(const name: string);

procedure push_data(const n: real);
function pop_data: real;
procedure push_word(const w: PForthWord);
function pop_word: PForthWord;

procedure refill;
procedure refill(const line_in: string);
procedure interpret;
procedure interpret(const code: TStringList);

implementation

procedure exec_proc(const w: PForthWord);
begin
	assert(w <> nil);
	w^.proc;
end;

procedure def_builtin(const name: string; const proc: TProcedure);
var
	w: PForthWord;
begin
	new(w);
	w^.exec := @exec_proc;
	w^.proc := proc;
	w^.code := nil;
	dictionary.items[name] := w;
end;

procedure push_data(const n: real);
begin
	if stack_pointer < length(stack) then
		begin
			stack[stack_pointer] := n;
			inc(stack_pointer);
		end
	else
		begin
			raise EForthError.create('Stack overflow');
		end
end;

function pop_data: real;
begin
	if stack_pointer > 0 then
		begin
			assert(length(stack) > 0);
			dec(stack_pointer);
			pop_data := stack[stack_pointer];
		end
	else
		begin
			raise EForthError.create('Stack underflow');
		end
end;

procedure push_word(const w: PForthWord);
begin
	if word_pointer < length(word_stack) then
		begin
			word_stack[word_pointer] := w;
			inc(word_pointer);
		end
	else
		begin
			raise EForthError.create('Word stack overflow');
		end
end;

function pop_word: PForthWord;
begin
	if word_pointer > 0 then
		begin
			assert(length(word_stack) > 0);
			dec(word_pointer);
			pop_word := word_stack[word_pointer];
		end
	else
		begin
			raise EForthError.create('Word stack underflow');
		end
end;

procedure exec_addr(const w: PForthWord);
begin
	push_word(w);
end;

procedure exec_data(const w: PForthWord);
begin
	assert(w <> nil);
	push_data(w^.data);
end;

procedure exec_code(const w: PForthWord);
begin
	assert(w <> nil);
	assert(w^.code <> nil);
	interpret(w^.code);
end;

procedure def_colon(const name: string);
var
	w: PForthWord;
begin
	new(w);
	w^.exec := @exec_code;
	w^.code := TStringList.create;
	dictionary.add(name, w); // Duplicate definitions would leak memory.
end;

procedure def_var(const name: string);
var
	w: PForthWord;
begin
	new(w);
	w^.exec := @exec_addr;
	w^.code := nil;
	dictionary.add(name, w); // Duplicate definitions would leak memory.
end;

procedure def_const(const name: string; const data: real);
var
	w: PForthWord;
begin
	new(w);
	w^.exec := @exec_code;
	w^.code := TStringList.create;
	w^.code.add(format('%g', [data]));
	dictionary.add(name, w); // Duplicate definitions would leak memory.
end;

procedure def_value(const name: string; const data: real);
var
	w: PForthWord;
begin
	new(w);
	w^.exec := @exec_data;
	w^.code := nil;
	w^.data := data;
	dictionary.add(name, w); // Duplicate definitions would leak memory.
end;

procedure execute(const name: string);
var
	w: PForthWord;
begin
	w := dictionary.items[name];
	if w <> nil then
		w^.exec(w)
	else
		push_data(strtofloat(name));
end;

procedure compile(const name: string);
begin
	if name = '\' then
		begin
			input_buffer.clear;
		end
	else if name <> parse_guard then
		begin
			parse_buffer.add(name);
		end
	else
		begin
			state := interpreting;
			execute(parse_guard);
			parse_guard := ''; { Technically not needed; helps debugging. }
		end;
end;

procedure refill(const line_in: string);
var
	i: integer;
begin
	input_buffer.clear;
	for i := 1 to wordcount(line_in, whitespace) do
		input_buffer.add(extractword(i, line_in, whitespace));
end;

procedure refill;
var
	line_in: string;
begin
	readln(line_in);
	refill(line_in);
end;

procedure interpret;
begin
	while input_buffer.count > 0 do
		case state of
			interpreting: execute(input_buffer.shift);
			buffering: compile(input_buffer.shift);
		end;
end;

procedure do_inc;
begin
	push_data(pop_data + 1)
end;

procedure do_dec;
begin
	push_data(pop_data - 1)
end;

procedure do_add;
begin
	push_data(pop_data + pop_data)
end;

procedure do_sub;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	push_data(a - b);
end;

procedure do_mul;
begin
	push_data(pop_data * pop_data)
end;

procedure do_div;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	push_data(a / b);
end;

procedure do_mod;
var
	a, b: integer;
begin
	b := trunc(pop_data);
	a := trunc(pop_data);
	push_data(a mod b);
end;

procedure do_negate;
begin
	push_data(-pop_data);
end;

procedure do_abs;
begin
	push_data(abs(pop_data));
end;

procedure do_min;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a < b then
		push_data(a)
	else
		push_data(b);
end;

procedure do_max;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a > b then
		push_data(a)
	else
		push_data(b);
end;

procedure do_round;
begin
	push_data(round(pop_data));
end;

procedure do_trunc;
begin
	push_data(trunc(pop_data));
end;

procedure do_random;
begin
	push_data(random);
end;

procedure do_sin;
begin
	push_data(sin(pop_data));
end;

procedure do_cos;
begin
	push_data(cos(pop_data));
end;

procedure do_sqrt;
begin
	push_data(sqrt(pop_data));
end;

procedure do_exp;
begin
	push_data(exp(pop_data));
end;

procedure do_ln;
begin
	push_data(ln(pop_data));
end;

procedure do_lt;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a < b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_lte;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a <= b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_eq;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a = b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_neq;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a <> b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_gt;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a > b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_gte;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if a >= b then
		push_data(1)
	else
		push_data(0);
end;

procedure do_and;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if (a <> 0) and (b <> 0) then
		push_data(1)
	else
		push_data(0);
end;

procedure do_or;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	if (a <> 0) or (b <> 0) then
		push_data(1)
	else
		push_data(0);
end;

procedure do_not;
begin
	if pop_data = 0 then
		push_data(1)
	else
		push_data(0);
end;

procedure do_swap;
var
	a, b: real;
begin
	b := pop_data;
	a := pop_data;
	push_data(b);
	push_data(a);
end;

procedure do_dup;
begin
	if stack_pointer > 0 then
		push_data(stack[stack_pointer - 1])
	else
		raise EForthError.create('Data stack empty in dup');
end;

procedure do_drop;
begin
	pop_data;
end;

procedure do_over;
begin
	if stack_pointer > 1 then
		push_data(stack[stack_pointer - 2])
	else
		raise EForthError.create('Data stack underflow in over');
end;

procedure do_2dup;
begin
	if stack_pointer > 1 then
		begin
			push_data(stack[stack_pointer - 2]);
			push_data(stack[stack_pointer - 2]);
		end
	else
		raise EForthError.create('Data stack underflow in 2dup');
end;

procedure do_rot;
var
	tmp: real;
begin
	if stack_pointer < 3 then
		raise EForthError.create('Data stack underflow in rot');
	tmp := stack[stack_pointer - 3];
	stack[stack_pointer - 3] := stack[stack_pointer - 2];
	stack[stack_pointer - 2] := stack[stack_pointer - 1];
	stack[stack_pointer - 1] := tmp;
end;

procedure do_nrot;
var
	tmp: real;
begin
	if stack_pointer < 3 then
		raise EForthError.create('Data stack underflow in -rot');
	tmp := stack[stack_pointer - 1];
	stack[stack_pointer - 1] := stack[stack_pointer - 2];
	stack[stack_pointer - 2] := stack[stack_pointer - 3];
	stack[stack_pointer - 3] := tmp;
end;

procedure do_pick;
var
	n: integer;
begin
	n := trunc(pop_data);
	if n < 1 then
		raise EForthError.create('Index too small in pick')
	else if stack_pointer < n then
		raise EForthError.create('Data stack underflow in pick')
	else
		push_data(stack[stack_pointer - n]);
end;

procedure do_print;
begin
	write(format('%g', [pop_data]), ' ');
end;

procedure do_cr;
begin
	writeln;
end;

procedure do_space;
begin
	write(' ');
end;

procedure do_spaces;
begin
	write(space(trunc(pop_data)));
end;

procedure do_emit;
begin
	write(chr(trunc(pop_data)));
end;

procedure do_char;
var
	w: string;
begin
	w := input_buffer.shift;
	if w = '' then
		raise EForthError.create('Text expected in char')
	else
		push_data(ord(w[1]));
end;

procedure do_ms;
begin
	sleep(trunc(pop_data));
end;

procedure do_pstack;
	var i: integer;
begin
	for i := 0 to stack_pointer - 1 do
		write(format('%g', [stack[i]]), ' ');
	writeln;
end;

procedure do_paren1;
begin
	parse_buffer.clear;
	state := buffering;
	parse_guard := ')';
end;

procedure do_paren2;
var
	i: string;
begin
	for i in parse_buffer do
		write(i, ' ');
end;

procedure do_bracket1;
begin
	parse_buffer.clear;
	state := buffering;
	parse_guard := ']';
end;

procedure do_bracket2;
begin
	code_buffer.addstrings(parse_buffer, true);
end;

procedure do_brace1;
begin
	parse_buffer.clear;
	state := buffering;
	parse_guard := '}';
end;

procedure do_brace2;
begin
	code_buffer.addstrings(parse_buffer, true);
end;

procedure do_rem1;
begin
	state := buffering;
	parse_guard := '*/';
end;

procedure do_rem2;
begin
	parse_buffer.clear;
end;

procedure do_variable;
var
	w: string;
begin
	w := input_buffer.shift;
	if w = '' then
		raise EForthError.create('Variable name expected');
	def_var(w);
	latest := w;
end;

procedure do_constant;
var
	w: string;
begin
	w := input_buffer.shift;
	if w = '' then
		raise EForthError.create('Constant name expected');
	def_const(w, pop_data);
	latest := w;
end;

procedure do_value;
var
	w: string;
begin
	w := input_buffer.shift;
	if w = '' then
		raise EForthError.create('Value name expected');
	def_value(w, pop_data);
	latest := w;
end;

procedure do_to;
var
	name: string;
	w: PForthWord;
begin
	name := input_buffer.shift;
	if name = '' then
		raise EForthError.create('Value name expected');
	w := dictionary.items[name];
	if w = nil then
		raise EForthError.create('Unknown word ' + name);
	w^.data := pop_data;
end;

procedure do_bang;
var
	w: PForthWord;
begin
	w := pop_word;
	w^.data := pop_data;
end;

procedure do_at;
var
	w: PForthWord;
begin
	w := pop_word;
	push_data(w^.data);
end;

procedure interpret(const code: TStringList); {< Exec buffer or colon def. }
var
	w: string;
begin
	for w in code do
		case state of
			interpreting: execute(w);
			buffering: compile(w);
		end;
end;

procedure do_iftrue;
var
	code: TStringList;
begin
	if pop_data <> 0 then
	begin
		code := TStringList.create;
		code.addstrings(code_buffer);
		try
			interpret(code);
		finally
			code.destroy;
		end;
	end; 
end;

procedure do_iffalse;
var
	code: TStringList;
begin
	if pop_data = 0 then
	begin
		code := TStringList.create;
		code.addstrings(code_buffer);
		try
			interpret(code);
		finally
			code.destroy;
		end;
	end; 
end;

procedure do_times;
var
	code: TStringList;
	i, limit: integer;
begin
	limit := trunc(pop_data);
	if limit > 0 then
	begin
		code := TStringList.create;
		code.addstrings(code_buffer);
		try
			for i := 1 to limit do
				interpret(code);
		finally
			code.destroy;
		end;
	end; 
end;

procedure do_while;
begin
	loop_buffer.addstrings(code_buffer, true);
	push_data(0);
end;

procedure do_until;
begin
	loop_buffer.addstrings(code_buffer, true);
	push_data(1);
end;

procedure do_repeat;
var
	body, test: TStringList;
	limit: real;
begin
	body := TStringList.create;
	test := TStringList.create;
	body.addstrings(loop_buffer);
	test.addstrings(code_buffer);
	try
		limit := pop_data;
		repeat
			interpret(body);
			interpret(test);
		until pop_data = limit;
	finally
		test.destroy;
		body.destroy;
	end;
end;

procedure do_colon;
var
	w: string;
begin
	w := input_buffer.shift;
	if w = '' then
		raise EForthError.create('Word name expected');
	def_colon(w);
	latest := w;
	parse_buffer.clear;
	state := buffering;
	parse_guard := ';';
end;

procedure do_semicolon;
var
	w: PForthWord;
begin
	w := dictionary.items[latest];
	if (w <> nil) and (w^.code <> nil) then
		w^.code.addstrings(parse_buffer, true)
	else
		raise EForthError.create('; without :');
end;

procedure do_find;
var
	name: string;
	w: PForthWord;
begin
	name := input_buffer.shift;
	if name = '' then
		raise EForthError.create('Word name expected');
	w := dictionary.items[name];
	if w = nil then
		raise EForthError.create('Unknown word ' + name);
	push_word(w);
end;

procedure do_execute;
var
	w: PForthWord;
begin
	w := pop_word;
	w^.exec(w);
end;

procedure do_backslash;
begin
	input_buffer.clear;
end;

procedure write_word(i: Pointer; const key: string; var iterate: boolean);
begin
	write(key, ' ');
end;

procedure do_words;
begin
	dictionary.iterate(@write_word);
	writeln;
end;

procedure do_see;
var
	name: string;
	w: PForthWord;
begin
	name := input_buffer.shift;
	if name = '' then
		raise EForthError.create('Word name expected in see');
	w := dictionary.items[name];
	if w = nil then
		raise EForthError.create('Unknown word ' + name)
	else if w^.code = nil then
		raise EForthError.create('(not a colon definition)');
	for name in w^.code do
		write(name, ' ');
	writeln;
end;

procedure do_forget;
var
	name: string;
	w: PForthWord;
begin
	name := input_buffer.shift;
	if name = '' then
		raise EForthError.create('Word name expected in forget');
	w := dictionary.items[name];
	if w = nil then
		raise EForthError.create('Unknown word ' + name);
	dictionary.delete(name);
	freeandnil(w^.code);
	dispose(w);
end;

procedure do_depth;
begin
	push_data(stack_pointer);
end;

procedure do_bye;
begin
	halt;
end;

procedure forget_word(i: Pointer; const key: string; var iterate: boolean);
var
	w: PForthWord;
begin
	w := i;
	freeandnil(w^.code);
	dispose(w);
end;

initialization
	dictionary := TFPDataHashTable.create;
	setlength(stack, 1024);
	stack_pointer := 0;
	setlength(word_stack, 1024);
	word_pointer := 0;
	input_buffer := TStringList.create;
	parse_buffer := TStringList.create;
	code_buffer := TStringList.create;
	loop_buffer := TStringList.create;
	parse_guard := '';
	state := interpreting;
	
	def_builtin('bye', @do_bye);

	def_builtin('depth', @do_depth);
	def_builtin('forget', @do_forget);
	def_builtin('see', @do_see);
	def_builtin('words', @do_words);
	def_builtin('.s', @do_pstack);

	def_builtin('''', @do_find);
	def_builtin('execute', @do_execute);
	
	def_builtin('constant', @do_constant);
	def_builtin('variable', @do_variable);
	def_builtin('value', @do_value);
	def_builtin('to', @do_to);
	def_builtin(':', @do_colon);
	def_builtin(';', @do_semicolon);
	def_builtin('\', @do_backslash);

	def_builtin('while', @do_while);
	def_builtin('until', @do_until);
	def_builtin('repeat', @do_repeat);

	def_builtin('times', @do_times);
	def_builtin('iftrue', @do_iftrue);
	def_builtin('iffalse', @do_iffalse);

	def_builtin('/*', @do_rem1);
	def_builtin('*/', @do_rem2);
	def_builtin('(', @do_paren1);
	def_builtin(')', @do_paren2);
	def_builtin('[', @do_bracket1);
	def_builtin(']', @do_bracket2);
	def_builtin('{', @do_brace1);
	def_builtin('}', @do_brace2);

	def_builtin('refill', @refill);
	def_builtin('interpret', @interpret);

	def_builtin('!', @do_bang);
	def_builtin('@', @do_at);	

	def_builtin('pick', @do_pick);
	def_builtin('rot', @do_rot);
	def_builtin('-rot', @do_nrot);
	def_builtin('over', @do_over);
	def_builtin('2dup', @do_2dup);
	def_builtin('swap', @do_swap);
	def_builtin('dup', @do_dup);
	def_builtin('drop', @do_drop);

	def_builtin('ms', @do_ms);

	def_builtin('char', @do_char);
	def_builtin('space', @do_space);
	def_builtin('spaces', @do_spaces);
	def_builtin('emit', @do_emit);
	def_builtin('cr', @do_cr);
	def_builtin('.', @do_print);

	def_builtin('and', @do_and);
	def_builtin('or', @do_or);
	def_builtin('not', @do_not);

	def_builtin('<', @do_lt);
	def_builtin('<=', @do_lte);
	def_builtin('=', @do_eq);
	def_builtin('<>', @do_neq);
	def_builtin('>', @do_gt);
	def_builtin('>=', @do_gte);

	def_builtin('round', @do_round);
	def_builtin('trunc', @do_trunc);
	def_builtin('random', @do_random);
	def_builtin('randomize', @randomize);
	def_builtin('sin', @do_sin);
	def_builtin('cos', @do_cos);
	def_builtin('sqrt', @do_sqrt);
	def_builtin('exp', @do_exp);
	def_builtin('ln', @do_ln);

	def_builtin('min', @do_min);
	def_builtin('max', @do_max);
	def_builtin('abs', @do_abs);
	def_builtin('negate', @do_negate);
	def_builtin('mod', @do_mod);
	def_builtin('/', @do_div);
	def_builtin('*', @do_mul);
	def_builtin('-', @do_sub);
	def_builtin('+', @do_add);

	def_builtin('1+', @do_inc);
	def_builtin('1-', @do_dec);

	def_const('true', 1);
	def_const('false', 0);
finalization
	dictionary.iterate(@forget_word);
	dictionary.destroy;
	input_buffer.destroy;
	parse_buffer.destroy;
	code_buffer.destroy;
	loop_buffer.destroy;
end.
