#ifndef BATCHBAS_HPP
#define BATCHBAS_HPP

#include <string>
#include <variant>
#include <vector>
#include <map>
#include <functional>
#include <optional>
#include <stdexcept>

#include <sstream>
#include <iostream>
#include <cctype>
#include <cmath>
#include <ctime>
#include <cstdlib>

namespace Basic {

const std::string library_version = "3.3";

using Num = double;
using Str = std::string;

typedef std::variant<Num, Str, bool> Value;
typedef std::vector<Value> Values;
typedef std::vector<Str> Lines;

struct Context;

typedef std::function<void(Context&)> Statement;
typedef std::function<Value(Context&, const Values&)> Function;

class ParseError : public std::runtime_error {
	public:
	explicit ParseError(const std::string& msg) : std::runtime_error(msg) { }
};

class RunError : public std::runtime_error {
	public:
	explicit RunError(const std::string& msg) : std::runtime_error(msg) { }
};

// Because <numbers> wasn't added until C++20.
const double pi = 3.141592653589793;
const double e = 2.718281828459045;

inline Num val(const Value& v) {
	switch (v.index()) {
		case 0: return std::get<Num>(v);
		case 1: return std::stod(std::get<Str>(v));
		case 2: return std::get<bool>(v) ? 1.0 : 0.0;
		default: throw RunError("Conversion error");
	}
}

inline Str str(const Value& v) {
	std::stringstream result;
	switch (v.index()) {
		case 0: // Can't use to_string() because it behaves like %f, not %g.
			result << std::get<Num>(v);
			return result.str();
		case 1: return std::get<Str>(v);
		case 2: return std::get<bool>(v) ? "true" : "false";
		default: throw RunError("Conversion error");
	}
}

inline bool tru(const Value& v) {
	switch (v.index()) {
		case 0: return std::get<Num>(v) != 0;
		case 1: return std::get<Str>(v).size();
		case 2: return std::get<bool>(v);
		default: throw RunError("Conversion error");
	}
}

inline void lower_case(Str& text) {
	for (auto& i: text)
		i = std::tolower(i);
}

inline void upper_case(Str& text) {
	for (auto& i: text)
		i = std::toupper(i);
}

inline bool starts_ident(char c) {
	return std::isalpha(c) || c == '_' || c == '$';
}

inline bool forms_ident(char c) {
	return std::isalnum(c) || c == '_' || c == '$';
}

struct Context {
	Str line;
	std::size_t cursor;
	Str token;
	
	Lines buffer;
	Str buf_to;
	Str expression; // Holds sub name, loop variable or condition.
	
	std::map<Str, Value> variables;
	std::map<Str, Statement> statements;
	std::map<Str, Function> functions;
	std::map<Str, Values> arrays;
	std::map<Str, Lines> subroutines;
	
	std::optional<bool> test_flag;
	Values data;
	std::size_t pointer = 0;
	
	bool option_explicit = false;
	std::size_t option_base = 0;
	
	void eval(const Str& code) {
		line = code;
		cursor = 0;
		if (at_end_of_line()) {
			return;
		} else if (!match_statement()) {
			// TO DO: add unknown subroutine support.
			throw ParseError("unknown statement");
		} else if (buf_to != "") {
			if (token != buf_to) {
				buffer.push_back(line);
			} else {
				buf_to = ""; // Do this first!!
				statements[token](*this);
			}
		} else {
			statements[token](*this);
			if (!at_end_of_line())
				throw ParseError("garbage at end of line");
		}
	}

	void skip_blanks() {
		while (cursor < line.size()
			&& std::isblank(line[cursor]))
				cursor++;
	}
	
	bool match_statement() {
		skip_blanks();
		const auto mark = cursor;
		while (cursor < line.size()
			&& std::isalpha(line[cursor]))
				cursor++;
		auto kw = line.substr(mark, cursor - mark);
		lower_case(kw);
		if (statements.count(kw) > 0) {
			token = kw;
			return true;
		} else {
			cursor = mark;
			return false;
		}
	}
	
	bool at_end_of_line() {
		skip_blanks();
		return (cursor >= line.size());
	}
	
	bool at_end_of_statement() {
		skip_blanks();
		return (cursor >= line.size() || line[cursor] == ':');
	}
	
	bool match_string() {
		skip_blanks();

		if (cursor >= line.size() || line[cursor] != '"')
			return false;
		
		const auto mark = cursor;

		do {
			cursor++;
			if (cursor >= line.size())
				throw ParseError("unclosed string");
		} while (line[cursor] != '"');
		
		cursor++; // Skip the closing double quote.
		
		// Save string value without the double quotes.
		token = line.substr(mark + 1, cursor - mark - 2);
		return true;
	}
	
	bool match(char c) {
		skip_blanks();
		if (cursor >= line.size() || line[cursor] != c) {
			return false;
		} else {
			token = line[cursor];
			cursor++;
			return true;
		}
	}
	
	bool match(const std::string& text) {
		skip_blanks();
		if (cursor >= line.size())
			return false;
		auto m = line.substr(cursor, text.size());
		lower_case(m);
		if (m == text) {
			cursor += m.size();
			token = m;
			return true;
		} else {
			return false;
		}
	}
	
	Value parse_expression() {
		auto result = parse_disjunction();
		while (match('&'))
			result = str(result) + str(parse_disjunction());
		return result; // Leave other expression results intact
	}
	
	Value parse_disjunction() {
		auto result = parse_conjunction();
		while (match("or")) {
			// C++ operators short-circuit; finish parsing first.
			const auto tmp = parse_conjunction();
			result = tru(result) || tru(tmp);
		}
		return result; // Leave other expression results intact
	}
	
	Value parse_conjunction() {
		auto result = parse_negation();
		while (match("and")) {
			// C++ operators short-circuit; finish parsing first.
			const auto tmp = parse_negation();
			result = tru(result) && tru(tmp);
		}
		return result; // Leave other expression results intact
	}
	
	Value parse_negation() {
		if (match("not"))
			return !tru(parse_comparison());
		else
			// Leave other expression results intact
			return parse_comparison();
	}
	
	Value parse_comparison() {
		const auto lhside = parse_arithmetic();
		if (match("="))
			return lhside == parse_arithmetic();
		else if (match("<>"))
			return lhside != parse_arithmetic();
		else if (match("<="))
			return lhside <= parse_arithmetic();
		else if (match("<"))
			return lhside < parse_arithmetic();
		else if (match(">="))
			return lhside >= parse_arithmetic();
		else if (match(">"))
			return lhside > parse_arithmetic();
		else
			return lhside;
	}
	
	Value parse_arithmetic() {
		auto result = parse_term();
		while (match('+') || match('-')) {
			if (token == "+")
				result = val(result) + val(parse_term());
			else if (token == "-")
				result = val(result) - val(parse_term());
		}
		return result;
	}
	
	Value parse_term() {
		auto result = parse_power();
		while (match('*') || match('/') || match("mod")) {
			if (token == "*")
				result = val(result) * val(parse_power());
			else if (token == "/")
				result = val(result) / val(parse_power());
			else if (token == "mod")
				result = std::fmod(
					val(result), val(parse_power()));
		}
		return result;
	}
	
	Value parse_power() {
		const auto result = parse_factor();
		if (match('^'))
			return std::pow(val(result), val(parse_power()));
		else
			return result;
	}
	
	Value parse_factor() {
		if (match("true"))
			return true;
		else if (match("false"))
			return false;
		else if (match_string())
			return token;
	
		Num signum = 0;

		if (match('-'))
			signum = -1;
		else if (match('+')) 
			signum = 1;
	
		if (match_number()) {
			if (signum)
				return std::stod(token) * signum;
			else
				return std::stod(token);
		} else if (match('(')) {
			const auto result = parse_expression();
			if (!match(')'))
				throw ParseError("unclosed parenthesis");
			else if (signum)
				return val(result) * signum;
			else
				return result;
		} else if (match_name()) {
			const auto name = token;
			if (match('(')) {
				Values args;
				if (!match(')')) {
					do {
						args.push_back(parse_expression());
					} while (match(','));
					if (!match(')'))
						throw ParseError(
							Str("unclosed parenthesis after ") + name);
				}
				if (functions.count(name) == 0) {
					throw RunError(Str("no such function: ") + name);
				} else if (signum) {
					const auto& fn = functions[name]; 
					return val(fn(*this, args)) * signum;
				} else {
					return functions[name](*this, args);
				}
			} else if (match('[')) {
				const std::size_t i = val(parse_expression());
				const auto& value = arrays.at(name).at(i - option_base);
				if (!match(']'))
					throw ParseError("unclosed bracket");
				else if (signum)
					return val(value) * signum;
				else
					return value;
			} else {
				if (option_explicit && variables.count(name) == 0)
					throw RunError(Str("variable not found: ") + name);
				
				if (signum)
					return val(variables[name]) * signum;
				else
					return variables[name];
			}
		} else {
			throw ParseError("expression expected");
		}
	}
	
	bool match_number() {
		skip_blanks();
		
		const auto mark = cursor;
		
		while (cursor < line.size()
			&& std::isdigit(line[cursor]))
				cursor++;
		
		if (cursor < line.size() && line[cursor] == '.') {
			cursor++;
			
			while (cursor < line.size()
				&& std::isdigit(line[cursor]))
					cursor++;
		}
		
		if (mark == cursor) {
			return false;
		} else {
			token = line.substr(mark, cursor - mark);
			return true;
		}
	}
	
	bool match_name() {
		skip_blanks();
		
		if (cursor >= line.size()
			|| !starts_ident(line[cursor]))
				return false;
			
		const auto mark = cursor;
		
		while (cursor < line.size()
			&& forms_ident(line[cursor]))
				cursor++;
			
		token = line.substr(mark, cursor - mark);
		lower_case(token);
		return true;
	}
	
	void run_statements() {
		while (match_statement()) {
			statements[token](*this);
			match(':');
		}
	}
};

inline Str ltrim(const Str& text) {
	std::size_t start = 0;
	while (start < text.size() && std::isspace(text[start]))
		start++;
	return text.substr(start);
}
inline Str rtrim(const Str& text) {
	std::size_t len = text.size();
	while (len > 0 && std::isspace(text[len - 1]))
		len--;
	return text.substr(0, len);
}
inline Str trim(const Str& text) {
	return ltrim(rtrim(text));
}

inline void load_builtins(Context& ctx) {
	ctx.statements["rem"] = [](Context& ctx) {
		ctx.cursor = ctx.line.size();
	};

	ctx.statements["print"] = [](Context& ctx) {
		while (!ctx.at_end_of_statement()) {
			std::cout << str(ctx.parse_expression());
			ctx.match(',') || ctx.match(';');
			if (ctx.token == ",")
				std::cout << "\t";
		}
		if (ctx.token != ";")
			std::cout << std::endl;
	};

	ctx.statements["let"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("identifier expected");
		
		const auto name = ctx.token;

		std::size_t index = 0;
		bool is_array = false;

		if (ctx.match('[')) {
			is_array = true;
			index = val(ctx.parse_expression());
			if (!ctx.match(']'))
				throw ParseError("unclosed bracket");
		}

		ctx.match('=');
		
		const auto value = ctx.parse_expression();
		
		if (is_array)
			ctx.arrays.at(name).at(index - ctx.option_base) = value;
		else
			ctx.variables[name] = value;
	};

	ctx.statements["input"] = [](Context& ctx) {
		if (ctx.match_string()) {
			std::cout << ctx.token;
			ctx.match(',');
		}
		
		std::vector<Str> vars;

		while (ctx.match_name()) {
			vars.push_back(ctx.token);
			ctx.match(',');
		}
		
		Str input;
		
		if (!getline(std::cin, input))
			throw ParseError("premature end of input");
		
		// Save parser state in case there are more statements.
		auto ln = ctx.line;
		auto cc = ctx.cursor;
		
		ctx.line = input;
		ctx.cursor = 0;
		
		for (const auto& i: vars) {
			if (ctx.match_number()) {
				ctx.variables[i] = std::stod(ctx.token);
				ctx.match(',');
			} else if (ctx.match_string()) {
				ctx.variables[i] = ctx.token;
				ctx.match(',');
			} else {
				ctx.line = ln;
				ctx.cursor = cc;
				throw ParseError("please enter a string or number");
			}
		}

		ctx.line = ln;
		ctx.cursor = cc;
	};

	ctx.statements["line"] = [](Context& ctx) {
		if (!ctx.match("input"))
			throw ParseError("INPUT expected");
		
		if (ctx.match_string()) {
			std::cout << ctx.token;
			ctx.match(',');
		}

		if (!ctx.match_name())
			throw ParseError("variable expected");
				
		Str input;
		
		if (getline(std::cin, input))
			ctx.variables[ctx.token] = input;
		else
			throw ParseError("premature end of input");
	};

	ctx.statements["if"] = [](Context& ctx) {
		const auto condition = val(ctx.parse_expression());
		
		if (!ctx.match("then"))
			throw ParseError("IF without THEN");
		else if (condition)
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["test"] = [](Context& ctx) {
		ctx.test_flag = (bool) val(ctx.parse_expression());
	};
		
	ctx.statements["iftrue"] = [](Context& ctx) {
		if (!ctx.test_flag.has_value())
			throw ParseError("IFTRUE without TEST");
		else if (ctx.test_flag.value())
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["iffalse"] = [](Context& ctx) {
		if (!ctx.test_flag.has_value())
			throw ParseError("IFFALSE without TEST");
		else if (!ctx.test_flag.value())
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["sub"] = [](Context& ctx) {
		if (ctx.match_name()) {
			ctx.expression = ctx.token;
			ctx.buffer.clear();
			ctx.buf_to = "return";
		} else {
			throw ParseError("subroutine name expected in SUB");
		}
	};

	ctx.statements["return"] = [](Context& ctx) {
		ctx.subroutines[ctx.expression] = ctx.buffer;
	};

	ctx.statements["call"] = [](Context& ctx) {
		if (ctx.match_name() && ctx.subroutines.count(ctx.token) > 0) {
			auto code = ctx.subroutines[ctx.token];
			for (const auto& i: code)
				ctx.eval(i); 
		} else {
			throw ParseError("subroutine name expected in CALL");
		}
	};

	ctx.statements["while"] = [](Context& ctx) {
		ctx.expression = ctx.line.substr(ctx.cursor);
		ctx.cursor = ctx.line.size();
		ctx.buffer.clear();
		ctx.buf_to = "wend";
	};

	ctx.statements["wend"] = [](Context& ctx) {
		const auto expr = ctx.expression;
		const auto code = ctx.buffer;
		do {
			ctx.line = expr;
			ctx.cursor = 0;
			if (val(ctx.parse_expression())) {
				for (const auto& i: code)
					ctx.eval(i);
			} else {
				break;
			}
		} while (true);
	};
	
	ctx.statements["for"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError(
				"variable name expected in FOR");
		
		ctx.expression = ctx.token;
		
		ctx.match('=');

		ctx.variables[ctx.expression] = ctx.parse_expression();
				
		ctx.match("to");

		ctx.variables["@limit"] = ctx.parse_expression();

		if (ctx.match("step"))
			ctx.variables["@step"] = ctx.parse_expression();
		else
			ctx.variables["@step"] = 1.0;
		
		ctx.buffer.clear();
		ctx.buf_to = "next";
	};
	
	ctx.statements["next"] = [](Context& ctx) {
		if (ctx.match_name() && ctx.token != ctx.expression)
			throw ParseError("loop variable mismatch in NEXT");
		
		const auto name = ctx.expression;
		const auto init = val(ctx.variables[name]);
		const auto limit = val(ctx.variables["@limit"]);
		const auto step = val(ctx.variables["@step"]);
		const auto code = ctx.buffer;
		
		if (step > 0) {
			for (auto i = init; i <= limit; i += step) { 
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else if (step < 0) {
			for (auto i = init; i >= limit; i += step) {
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else {
			throw RunError("FOR loop with a STEP of zero");
		}
		
		ctx.variables.erase("@limit");
		ctx.variables.erase("@step");
	};

	ctx.statements["repeat"] = [](Context& ctx) {
		ctx.buffer.clear();
		ctx.buf_to = "until";
	};

	ctx.statements["until"] = [](Context& ctx) {
		const auto expr = ctx.line.substr(ctx.cursor);
		const auto code = ctx.buffer;
		do {
			for (const auto& i: code)
				ctx.eval(i);
			ctx.line = expr;
			ctx.cursor = 0;
		} while (!val(ctx.parse_expression()));
	};
	
	ctx.statements["do"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("variable name expected in DO");
				
		ctx.expression = ctx.token;
		
		ctx.match('=');

		ctx.variables[ctx.expression] = ctx.parse_expression();
				
		ctx.match(',');

		ctx.variables["@limit"] = ctx.parse_expression();

		if (ctx.match(','))
			ctx.variables["@step"] = ctx.parse_expression();
		else
			ctx.variables["@step"] = 1.0;
		
		ctx.buffer.clear();
		ctx.buf_to = "done";
	};
	
	ctx.statements["done"] = [](Context& ctx) {
		const auto name = ctx.expression;
		const auto init = val(ctx.variables[name]);
		const auto limit = val(ctx.variables["@limit"]);
		const auto step = val(ctx.variables["@step"]);
		const auto code = ctx.buffer;
		
		if (step > 0) {
			for (auto i = init; i <= limit; i += step) { 
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else if (step < 0) {
			for (auto i = init; i >= limit; i += step) {
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else {
			throw RunError("DO loop with a step of zero");
		}
		
		ctx.variables.erase("@limit");
		ctx.variables.erase("@step");
	};
	
	ctx.statements["data"] = [](Context& ctx) {
		do {
			ctx.data.push_back(ctx.parse_expression());
		} while (ctx.match(','));
	};
	
	ctx.statements["read"] = [](Context& ctx) {
		while (ctx.match_name()) {
			if (ctx.pointer >= ctx.data.size())
				throw RunError("READ past the end of DATA");
			ctx.variables[ctx.token] = ctx.data[ctx.pointer];
			ctx.pointer++;
			ctx.match(',');
		}
	};
	
	ctx.statements["restore"] = [](Context& ctx) {
		std::size_t pos = 0;
		if (!ctx.at_end_of_statement())
			pos = val(ctx.parse_expression());
		if (pos >= ctx.data.size())
			throw RunError("RESTORE out of DATA range");
		else
			ctx.pointer = pos;
	};
	
	ctx.statements["randomize"] = [](Context& ctx) {
		if (ctx.at_end_of_statement())
			std::srand(std::time(nullptr));
		else
			std::srand(val(ctx.parse_expression()));
	};
	
	ctx.statements["poke"] = [](Context& ctx) {
		std::size_t pos = val(ctx.parse_expression());
		if (pos >= ctx.data.size())
			throw RunError("POKE out of DATA range");
		else if (!ctx.match(','))
			throw ParseError("data argument expected");
		else
			ctx.data[pos] = ctx.parse_expression();
	};
	
	ctx.statements["erase"] = [](Context& ctx) {
		while (ctx.match_name()) {
			ctx.variables.erase(ctx.token);
			ctx.arrays.erase(ctx.token);
			ctx.match(',');
		}
	};
	
	ctx.statements["option"] = [](Context& ctx) {
		if (ctx.match("explicit"))
			ctx.option_explicit = true;
		else if (ctx.match("base"))
			ctx.option_base = std::abs(val(ctx.parse_expression()));
		else
			throw ParseError("option name expected");
	};
	
	ctx.statements["on"] = [](Context& ctx) {
		auto choice = val(ctx.parse_expression());
		
		ctx.match("gosub");
		
		std::vector<std::string> names;
		
		while (ctx.match_name()) {
			names.push_back(ctx.token);
			ctx.match(',');
		}
		
		if (1 <= choice && choice <= names.size()) {
			const auto& name = names[choice - 1];
			const auto& code = ctx.subroutines.at(name);
			for (const auto& i: code)
				ctx.eval(i); 
		}
	};
	
	ctx.statements["dim"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("array name expected in DIM");
				
		const auto name = ctx.token;
		
		if (ctx.arrays.count(name) > 0) {
			throw RunError("array already exists");
		} else if (ctx.match('[')) {
			const std::size_t size = val(ctx.parse_expression());
			if (ctx.match(']'))
				ctx.arrays[name].resize(size);
			else
				throw ParseError("unclosed bracket");
		} else {
			throw ParseError("bracket expected in DIM");
		}
	};
	
	ctx.statements["redim"] = [](Context& ctx) {
		ctx.match("preserve");
	
		if (!ctx.match_name())
			throw ParseError("array name expected in REDIM");
				
		const auto name = ctx.token;
		
		if (ctx.arrays.count(name) == 0) {
			throw RunError("array doesn't exist");
		} else if (ctx.match('[')) {
			const std::size_t size = val(ctx.parse_expression());
			if (ctx.match(']'))
				ctx.arrays[name].resize(size);
			else
				throw ParseError("unclosed bracket");
		} else {
			throw ParseError("bracket expected in REDIM");
		}
	};

	ctx.statements["swap"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("variable name expected in SWAP");
		const auto v1 = ctx.token;
		
		ctx.match(',');
		if (!ctx.match_name())
			throw ParseError("variable name expected in SWAP");
		const auto v2 = ctx.token;
		
		if (ctx.option_explicit)
			std::swap(ctx.variables.at(v1), ctx.variables.at(v2));
		else
			std::swap(ctx.variables[v1], ctx.variables[v2]);
	};
	
	ctx.statements["eval"] = [](Context& ctx) {
		Str code = str(ctx.parse_expression());
		
		// Save parser state in case there are more statements.
		auto ln = ctx.line;
		auto cc = ctx.cursor;
		
		ctx.line = code;
		ctx.cursor = 0;
		ctx.run_statements();
		ctx.line = ln;
		ctx.cursor = cc;
	};
	
	ctx.functions["abs"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::abs(val(args[0]));
		else
			throw ParseError("ABS expects exactly one argument");
	};
	
	ctx.functions["int"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::floor(val(args[0]));
		else
			throw ParseError("INT expects exactly one argument");
	};
	
	ctx.functions["sqr"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::sqrt(val(args[0]));
		else
			throw ParseError("SQR expects exactly one argument");
	};
	
	ctx.functions["sin"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::sin(val(args[0]));
		else
			throw ParseError("SIN expects exactly one argument");
	};
	
	ctx.functions["cos"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::cos(val(args[0]));
		else
			throw ParseError("COS expects exactly one argument");
	};
	
	ctx.functions["pi"] = [](Context& ctx, const Values& args) {
		return pi;
	};
	
	ctx.functions["e"] = [](Context& ctx, const Values& args) {
		return e;
	};
	
	ctx.functions["hypot"] = [](Context& ctx, const Values& args) {
		switch (args.size()) {
			case 2:
				return std::hypot(val(args[0]), val(args[1]));
			case 3:
				return std::hypot(val(args[0]), val(args[1]), val(args[2]));
			default:
				throw ParseError("hypot takes two or three arguments");
		}
	};

	ctx.functions["rad"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return val(args[0]) * pi / 180;
		else
			throw ParseError("RAD expects exactly one argument");
	};

	ctx.functions["deg"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return val(args[0]) * 180 / pi;
		else
			throw ParseError("DEG expects exactly one argument");
	};
	
	ctx.functions["min"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return std::min(args[0], args[1]);
		else
			throw ParseError("MIN expects exactly two arguments");
	};
	
	ctx.functions["max"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return std::max(args[0], args[1]);
		else
			throw ParseError("MAX expects exactly two arguments");
	};
	
	ctx.functions["exp"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::exp(val(args[0]));
		else
			throw ParseError("EXP expects exactly one argument");
	};
	
	ctx.functions["log"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::log(val(args[0]));
		else
			throw ParseError("LOG expects exactly one argument");
	};
	
	ctx.functions["timer"] = [](Context& ctx, const Values& args) {
		return (double)std::clock();
	};
	
	ctx.functions["rnd"] = [](Context& ctx, const Values& args) {
		return (double) std::rand() / ((double) RAND_MAX + 1);
	};
	
	ctx.functions["peek"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return ctx.data.at(val(args[0]));
		else
			throw ParseError("PEEK expects exactly one argument");
	};
	
	ctx.functions["sgn"] = [](Context& ctx, const Values& args) {
		if (args.size() != 1)
			throw ParseError("SGN expects exactly one argument");
		else if (val(args[0]) < 0)
			return -1.0;
		else if (val(args[0]) > 0)
			return 1.0;
		else
			return 0.0;
	};

	ctx.functions["len"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return (double)str(args[0]).size();
		else
			throw ParseError("LEN expects exactly one argument");
	};
	ctx.functions["val"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return val(args[0]);
		else
			throw ParseError("VAL expects exactly one argument");
	};
	ctx.functions["asc"] = [](Context& ctx, const Values& args) {
		if (args.size() != 1)
			throw ParseError("ASC expects exactly one argument");
		else
			return (Num) str(args[0]).at(0);
	};
	
	ctx.functions["str$"] = [](Context& ctx, const Values& args) {
		Str result;
		for (const auto& i: args)
			result += str(i);
		return result;
	};
	ctx.functions["chr$"] = [](Context& ctx, const Values& args) {
		Str result;
		for (const auto& i: args)
			result += (char) val(i);
		return result;
	};
	ctx.functions["space$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return Str(val(args[0]), ' ');
		else
			throw ParseError("SPACE$ expects exactly one argument");
	};

	ctx.functions["lcase$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			auto tmp = str(args[0]);
			lower_case(tmp);
			return tmp;
		} else {
			throw ParseError("LCASE$ expects exactly one argument");
		}
	};
	ctx.functions["ucase$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			auto tmp = str(args[0]);
			upper_case(tmp);
			return tmp;
		} else {
			throw ParseError("LCASE$ expects exactly one argument");
		}
	};
	ctx.functions["ltrim$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			return ltrim(str(args[0]));
		} else {
			throw ParseError("LTRIM expects exactly one argument");
		}
	};
	ctx.functions["rtrim$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			return rtrim(str(args[0]));
		} else {
			throw ParseError("LTRIM expects exactly one argument");
		}
	};
	ctx.functions["trim$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			return trim(str(args[0]));
		} else {
			throw ParseError("trim expects exactly one argument");
		}
	};

	ctx.functions["left$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return str(args[0]).substr(0, val(args[1]));
		else
			throw ParseError("LEFT$ expects exactly two arguments");
	};
	ctx.functions["right$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2) {
			auto tmp = str(args[0]);
			return tmp.substr(tmp.size() - val(args[1]));
		} else {
			throw ParseError("RIGHT$ expects exactly two arguments");
		}
	};
	ctx.functions["mid$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return str(args[0]).substr(val(args[1]));
		else if (args.size() == 3)
			return str(args[0]).substr(
				val(args[1]), val(args[2]));
		else
			throw ParseError("MID$ expects two or three arguments");
	};
	
	ctx.functions["iif"] = [](Context& ctx, const Values& args) {
		if (args.size() == 3) {
			return tru(args[0]) ? args[1] : args[2];
		} else {
			throw ParseError("IIF takes exactly three arguments");
		}
	};
	ctx.functions["expr"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			Str code = str(args[0]);
			
			// Save parser state in case there are more statements.
			auto ln = ctx.line;
			auto cc = ctx.cursor;
			
			ctx.line = code;
			ctx.cursor = 0;
			
			auto result = ctx.parse_expression();
			
			ctx.line = ln;
			ctx.cursor = cc;
			
			return result;
		} else {
			throw ParseError("EXPR expects exactly one argument");
		}
	};
}

}

#endif
