#include <iostream>
#include <fstream>
#include <cstdlib>

#include "batchbas.hpp"

using Ctx = Basic::Context;

const std::string interpreter_version = "2025-10-18";

const std::string help_text[] = {
	" [ -v | -h | -e CODE | -l FILE | PROGRAM ]",
	"",
	"Arguments:",
	"    PROGRAM:       run given program",
	"",
	"Options:",
	"    -h, --help:    show this message and quit",
	"    -v, --version: show version and quit",
	"    -e, --eval:    evaluate given CODE and quit",
	"    -l, --load:    load given FILE and go interactive",
	"    -q, --quiet:   suppress banner and prompt while interactive"
};

bool merge(const Basic::Str& filename, Ctx& ctx) {
	std::ifstream source(filename);
	if (source.is_open()) {
		Basic::Str line;
		while(std::getline(source, line))
			if (line.length() > 0 && line[0] != '\'')
				ctx.eval(line);
		return true;
	} else {
		return false;
	}
}

bool try_to_merge(const Basic::Str& filename, Ctx& ctx) {
	try {
		return merge(filename, ctx);
	} catch (const Basic::ParseError& e) {
		std::cerr << "Parsing error in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const Basic::RunError& e) {
		std::cerr << "Runtime error in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const std::invalid_argument& e) {
		std::cerr << "Invalid argument in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const std::out_of_range& e) {
		std::cerr << "Out of range in line:\n"
			<< ctx.line << "\n" << std::endl; // No need for details here.
		return false;
	}
}

void do_merge(Ctx& ctx) {
	if (!ctx.match_string())
		throw Basic::ParseError("filename expected");
	else if (!merge(ctx.token, ctx))
		throw Basic::RunError("can't open file");
};

void do_list(Ctx& ctx) {
	if (!ctx.match_name()) {
		for (const auto& i: ctx.subroutines) {
			std::cout << i.first << std::endl;
		}
	} else if (ctx.subroutines.count(ctx.token) > 0) {
		const auto& s = ctx.subroutines[ctx.token];
		for (const auto& i: s)
			std::cout << i << "\n";
		std::cout << std::endl;
	} else {
		throw Basic::RunError("subroutine not found");
	}
}

void do_delete(Ctx& ctx) {
	if (!ctx.match_name())
		throw Basic::ParseError("subroutine name expected");
	else
		ctx.subroutines.erase(ctx.token);
}

void do_new(Ctx& ctx) {
	ctx.variables.clear();
	ctx.data.clear();
	ctx.subroutines.clear();
}

Basic::Str repr(const Basic::Value& data) {
	switch (data.index()) {
		case 0: return std::to_string(std::get<Basic::Num>(data));
		case 1: return "\"" + std::get<Basic::Str>(data) + "\"";
		case 2: return std::get<bool>(data) ? "true" : "false";
		default: throw Basic::RunError("Conversion error");
	}
}

void do_save(Ctx& ctx) {
	if (!ctx.match_string())
		throw Basic::ParseError("filename expected");
	std::ofstream dest(ctx.token);
	if (dest.is_open()) {
		for (std::size_t i = 0; i < ctx.data.size(); i++) {
			if (i % 10 == 0)
				dest << "data ";
			dest << repr(ctx.data[i]);
			if (i % 10 == 9)
				dest << '\n';
			else
				dest << ", ";
		}
		if (ctx.data.size() > 0)
			dest << std::endl;
		for (const auto& i: ctx.variables)
			dest << "let " << i.first
				<< " = " << repr(i.second) << '\n';
		if (ctx.variables.size() > 0)
			dest << std::endl;
		for (const auto& i: ctx.subroutines) {
			dest << "sub " << i.first << '\n';
			for (const auto& j: i.second)
				dest << j << '\n';
			dest << "return" << std::endl;
		}
	} else {
		throw Basic::RunError(std::string("can't open file: ") + ctx.token);
	}
}

int main(int argc, char* argv[]) {
	using namespace Basic;

	Context ctx;
	load_builtins(ctx);

	ctx.functions["environ$"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1) {
			return std::getenv(Basic::str(args[0]).c_str());
		} else {
			throw ParseError("environ$ takes exactly one argument");
		}
	};
	ctx.statements["merge"] = do_merge;
	ctx.statements["quit"] = [](Context& ctx) { std::exit(0); };

	std::vector<Str> args;
	args.reserve(argc);
	for (int i = 0; i < argc; i++)
		args.push_back(argv[i]);

	bool quiet = false;
	bool interactive = false;

	if (args.size() < 2) {
		interactive = true;
	} else if (args[1] == "-v" || args[1] == "--version") {
		std::cout << "Batch Basic " << library_version;
		std::cout << " / " << interpreter_version << std::endl;
		return 0;
	} else if (args[1] == "-h" || args[1] == "--help") {
		std::cout << "Usage: " << args[0];
		for (const auto& i: help_text)
			std::cout << i << '\n';
		std::cout << std::flush;
		return 0;
	} else if (args[1] == "-e" || args[1] == "--eval") {
		if (args.size() > 2) {
			ctx.line = args[2];
			ctx.cursor = 0; 
			ctx.run_statements();
		} else {
			std::cerr << "Missing argument to " << args[1] << std::endl;
			return 1;
		}
	} else if (args[1] == "-l" || args[1] == "--load") {
		if (args.size() < 3) {
			std::cerr << "Missing argument to " << args[1] << std::endl;
			return 1;
		}
		if (try_to_merge(args[2], ctx))
			std::cout << "Loaded " << args[2] << std::endl;
		else
			std::cerr << "Failed to load " << args[2] << std::endl;
		interactive = true;
	} else if (args[1] == "-q" || args[1] == "--quiet") {
		quiet = true;
		interactive = true;
	} else {
		Values basic_args;
		basic_args.reserve(args.size() - 1);
		for (int i = 1; i < args.size(); i++)
			basic_args.emplace_back(args[i]);
		ctx.arrays["argv"] = basic_args;
		ctx.variables["argc"] = (double)basic_args.size();
		if (!try_to_merge(args[1], ctx)) {
			return 2;
		} else if (ctx.subroutines.count("main") > 0) {
			auto code = ctx.subroutines["main"];
			for (const auto& i: code)
				ctx.eval(i); 
		}
	}
	
	if (!interactive) return 0;

	ctx.statements["clear"] = [](Context& ctx) {
		ctx.variables.clear();
	};
	ctx.statements["load"] = [](Context& ctx) {
		do_new(ctx);
		do_merge(ctx);
	};
	ctx.statements["new"] = do_new;
	ctx.statements["delete"] = do_delete;
	ctx.statements["list"] = do_list;
	
	ctx.statements["save"] = do_save;

	if (!quiet) {
		std::cout << "Batch Basic " << library_version;
		std::cout << " / " << interpreter_version << '\n';
		std::cout << ctx.statements.size() << " statements and ";
		std::cout << ctx.functions.size() << " functions defined.\n";
		std::cout << "Enter code or QUIT to exit:\n";
		std::cout << "> " << std::flush;
	}

	Str line;
	
	while(std::getline(std::cin, line)) {
		try {
			if (line.length() > 0 && line[0] != '\'')
				ctx.eval(line);
			if (ctx.buf_to == "" && !quiet)
				std::cout << "OK\n";
		} catch (const Basic::ParseError& e) {
			std::cerr << "Parsing error in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const Basic::RunError& e) {
			std::cerr << "Runtime error in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const std::invalid_argument& e) {
			std::cerr << "Invalid argument in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const std::out_of_range& e) {
			std::cerr << "Out of range in line:\n"
				<< line << "\n" << std::endl; // No need for details here.
		}
		if (!quiet)
			std::cout << (ctx.buf_to == "" ? "> " : "... ") << std::flush;
	}
	
	return 0;
}
