#ifndef BATCHBAS_HPP
#define BATCHBAS_HPP

#include <functional>
#include <vector>
#include <map>
#include <string>
#include <stdexcept>
#include <cctype>
#include <iostream>
#include <cmath>
#include <ctime>

namespace basic {

const std::string version_number = "2.7 beta";

typedef std::vector<double> Values;
typedef std::vector<std::string> Lines;

struct Context;

typedef std::function<void(Context&)> Statement;
typedef std::function<double(Context&, const Values&)> Function;

class ParseError : public std::runtime_error {
	public:
	explicit ParseError(const std::string& msg) : std::runtime_error(msg) { }
};

class RunError : public std::runtime_error {
	public:
	explicit RunError(const std::string& msg) : std::runtime_error(msg) { }
};

// Because <numbers> wasn't added until C++20.
const double pi = 3.141592653589793;
const double e = 2.718281828459045;

inline void normalize_case(std::string& text) {
	for (auto& i: text)
		i = std::tolower(i);
}

inline bool starts_ident(char c) {
	return std::isalpha(c) || c == '_';
}

inline bool forms_ident(char c) {
	return std::isalnum(c) || c == '_';
}

struct Context {
	std::string line;
	std::size_t cursor;
	std::string token;
	
	std::vector<std::string> buffer;
	std::string buf_to;
	std::string expression; // Holds sub name, loop variable or condition.
	
	std::map<std::string, double> variables;
	std::map<std::string, Statement> statements;
	std::map<std::string, Function> functions;
	std::map<std::string, Values> arrays;
	std::map<std::string, Lines> subroutines;
	
	double test_flag = NAN;
	Values data;
	std::size_t pointer = 0;
	
	bool option_explicit = false;
	std::size_t option_base = 0;
	
	void eval(const std::string& code) {
		line = code;
		cursor = 0;
		if (at_end_of_line()) {
			return;
		} else if (!match_statement()) {
			// TO DO: add unknown subroutine support.
			throw ParseError("unknown statement");
		} else if (buf_to != "") {
			if (token != buf_to) {
				buffer.push_back(line);
			} else {
				buf_to = ""; // Do this first!!
				statements[token](*this);
			}
		} else {
			statements[token](*this);
			if (!at_end_of_line())
				throw ParseError("garbage at end of line");
		}
	}

	void skip_blanks() {
		while (cursor < line.size()
			&& std::isblank(line[cursor]))
				cursor++;
	}
	
	bool match_statement() {
		skip_blanks();
		const auto mark = cursor;
		while (cursor < line.size()
			&& std::isalpha(line[cursor]))
				cursor++;
		auto kw = line.substr(mark, cursor - mark);
		normalize_case(kw);
		if (statements.count(kw) > 0) {
			token = kw;
			return true;
		} else {
			cursor = mark;
			return false;
		}
	}
	
	bool at_end_of_line() {
		skip_blanks();
		return (cursor >= line.size());
	}
	
	bool at_end_of_statement() {
		skip_blanks();
		return (cursor >= line.size() || line[cursor] == ':');
	}
	
	bool match_string() {
		skip_blanks();

		if (cursor >= line.size() || line[cursor] != '"')
			return false;
		
		const auto mark = cursor;

		do {
			cursor++;
			if (cursor >= line.size())
				throw ParseError("unclosed string");
		} while (line[cursor] != '"');
		
		cursor++; // Skip the closing double quote.
		
		// Save string value without the double quotes.
		token = line.substr(mark + 1, cursor - mark - 2);
		return true;
	}
	
	bool match(char c) {
		skip_blanks();
		if (cursor >= line.size() || line[cursor] != c) {
			return false;
		} else {
			token = line[cursor];
			cursor++;
			return true;
		}
	}
	
	bool match(const std::string& text) {
		skip_blanks();
		if (cursor >= line.size())
			return false;
		auto m = line.substr(cursor, text.size());
		normalize_case(m);
		if (m == text) {
			cursor += m.size();
			token = m;
			return true;
		} else {
			return false;
		}
	}
	
	double parse_expression() {
		return parse_disjunction();
	}
	
	double parse_disjunction() {
		auto result = parse_conjunction();
		while (match("or")) {
			auto tmp = parse_conjunction();
			result = (result || tmp) ? -1 : 0;
		}
		return result; // Leave purely arithmetic results intact
	}
	
	double parse_conjunction() {
		auto result = parse_negation();
		while (match("and")) {
			auto tmp = parse_negation();
			result = (result && tmp) ? -1 : 0;
		}
		return result; // Leave purely arithmetic results intact
	}
	
	double parse_negation() {
		if (match("not"))
			return parse_comparison() ? 0 : -1;
		else
			// Leave purely arithmetic results intact
			return parse_comparison();
	}
	
	double parse_comparison() {
		const auto lhside = parse_arithmetic();
		if (match("="))
			return lhside == parse_arithmetic() ? -1 : 0;
		else if (match("<>"))
			return lhside != parse_arithmetic() ? -1 : 0;
		else if (match("<="))
			return lhside <= parse_arithmetic() ? -1 : 0;
		else if (match("<"))
			return lhside < parse_arithmetic() ? -1 : 0;
		else if (match(">="))
			return lhside >= parse_arithmetic() ? -1 : 0;
		else if (match(">"))
			return lhside > parse_arithmetic() ? -1 : 0;
		else
			return lhside;
	}
	
	double parse_arithmetic() {
		auto result = parse_term();
		while (match('+') || match('-')) {
			if (token == "+")
				result += parse_term();
			else if (token == "-")
				result -= parse_term();
		}
		return result;
	}
	
	double parse_term() {
		auto result = parse_power();
		while (match('*') || match('/') || match("mod")) {
			if (token == "*")
				result *= parse_power();
			else if (token == "/")
				result /= parse_power();
			else if (token == "mod")
				result = std::fmod(result, parse_power());
		}
		return result;
	}
	
	double parse_power() {
		const auto result = parse_factor();
		if (match('^'))
			return std::pow(result, parse_power());
		else
			return result;
	}
	
	double parse_factor() {
		if (match("true"))
			return -1;
		else if (match("false"))
			return 0;
	
		double signum = 1;

		if (match('-'))
			signum = -1;
		else if (match('+')) 
			signum = 1;
	
		if (match_number()) {
			return std::stod(token) * signum;
		} else if (match('(')) {
			const auto result = parse_expression();
			if (match(')'))
				return result * signum;
			else
				throw ParseError("unclosed parenthesis");
		} else if (match_name()) {
			const auto name = token;
			if (match('(')) {
				Values args;
				while (!match(')')) {
					args.push_back(parse_expression());
					match(',');
				}
				return functions[name](*this, args) * signum;
			} else if (match('[')) {
				const std::size_t index = parse_expression();
				if (match(']'))
					return arrays.at(name).at(index - option_base)
						* signum;
				else
					throw ParseError("unclosed bracket");
			} else if (option_explicit) {
				if (variables.count(name) > 0)
					return variables[name] * signum;
				else
					throw RunError(std::string("variable not found: ") + name);
			} else {
				return variables[name] * signum;
			}
		} else {
			throw ParseError("value or expression expected");
		}
	}
	
	bool match_number() {
		skip_blanks();
		
		const auto mark = cursor;
		
		while (cursor < line.size()
			&& std::isdigit(line[cursor]))
				cursor++;
		
		if (cursor < line.size() && line[cursor] == '.') {
			cursor++;
			
			while (cursor < line.size()
				&& std::isdigit(line[cursor]))
					cursor++;
		}
		
		if (mark == cursor) {
			return false;
		} else {
			token = line.substr(mark, cursor - mark);
			return true;
		}
	}
	
	bool match_name() {
		skip_blanks();
		
		if (cursor >= line.size()
			|| !starts_ident(line[cursor]))
				return false;
			
		const auto mark = cursor;
		
		while (cursor < line.size()
			&& forms_ident(line[cursor]))
				cursor++;
			
		token = line.substr(mark, cursor - mark);
		normalize_case(token);
		return true;
	}
	
	void run_statements() {
		while (match_statement()) {
			statements[token](*this);
			match(':');
		}
	}
};

inline void load_builtins(Context& ctx) {
	ctx.statements["rem"] = [](Context& ctx) {
		ctx.cursor = ctx.line.size();
	};

	ctx.statements["print"] = [](Context& ctx) {
		while (!ctx.at_end_of_statement()) {
			if (ctx.match_string())
				std::cout << ctx.token;
			else if (ctx.match("chr$"))
				std::cout << (char) ctx.parse_expression();
			else
				std::cout << ctx.parse_expression();
			ctx.match(',') || ctx.match(';');
			if (ctx.token == ",")
				std::cout << "\t";
		}
		if (ctx.token != ";")
			std::cout << std::endl;
	};

	ctx.statements["let"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("identifier expected");
		
		const auto name = ctx.token;

		std::size_t index = 0;
		bool is_array = false;

		if (ctx.match('[')) {
			is_array = true;
			index = ctx.parse_expression();
			if (!ctx.match(']'))
				throw ParseError("unclosed bracket");
		}

		ctx.match('=');
		
		const auto value = ctx.parse_expression();
		
		if (is_array)
			ctx.arrays.at(name).at(index - ctx.option_base) = value;
		else
			ctx.variables[name] = value;
	};

	ctx.statements["input"] = [](Context& ctx) {
		if (ctx.match_string()) {
			std::cout << ctx.token;
			ctx.match(',');
		}
		
		std::vector<std::string> vars;

		while (ctx.match_name()) {
			vars.push_back(ctx.token);
			ctx.match(',');
		}
		
		std::string input;
		
		if (!getline(std::cin, input))
			throw ParseError("premature end of input");
		
		// Save parser state in case there are more statements.
		auto ln = ctx.line;
		auto cc = ctx.cursor;
		
		ctx.line = input;
		ctx.cursor = 0;
		
		for (const auto& i: vars) {
			if (ctx.match_number()) {
				ctx.variables[i] = std::stod(ctx.token);
				ctx.match(',');
			} else {
				ctx.line = ln;
				ctx.cursor = cc;
				throw ParseError("please enter a number");
			}
		}

		ctx.line = ln;
		ctx.cursor = cc;
	};

	ctx.statements["if"] = [](Context& ctx) {
		const auto condition = ctx.parse_expression();
		
		if (!ctx.match("then"))
			throw ParseError("IF without THEN");
		else if (condition)
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["test"] = [](Context& ctx) {
		ctx.test_flag = ctx.parse_expression();
	};
		
	ctx.statements["iftrue"] = [](Context& ctx) {
		if (!std::isnan(ctx.test_flag) && ctx.test_flag)
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["iffalse"] = [](Context& ctx) {
		if (!std::isnan(ctx.test_flag) && !ctx.test_flag)
			ctx.run_statements();
		else
			ctx.cursor = ctx.line.size();
	};
	
	ctx.statements["sub"] = [](Context& ctx) {
		if (ctx.match_name()) {
			ctx.expression = ctx.token;
			ctx.buffer.clear();
			ctx.buf_to = "return";
		} else {
			throw ParseError("subroutine name expected in SUB");
		}
	};
	
	ctx.statements["return"] = [](Context& ctx) {
		ctx.subroutines[ctx.expression] = ctx.buffer;
	};
	
	ctx.statements["call"] = [](Context& ctx) {
		if (ctx.match_name() && ctx.subroutines.count(ctx.token) > 0) {
			const auto& code = ctx.subroutines[ctx.token];
			for (const auto& i: code)
				ctx.eval(i); 
		} else {
			throw ParseError("subroutine name expected in CALL");
		}
	};

	ctx.statements["while"] = [](Context& ctx) {
		ctx.expression = ctx.line.substr(ctx.cursor);
		ctx.cursor = ctx.line.size();
		ctx.buffer.clear();
		ctx.buf_to = "wend";
	};

	ctx.statements["wend"] = [](Context& ctx) {
		const auto expr = ctx.expression;
		const auto code = ctx.buffer;
		do {
			ctx.line = expr;
			ctx.cursor = 0;
			if (ctx.parse_expression()) {
				for (const auto& i: code)
					ctx.eval(i);
			} else {
				break;
			}
		} while (true);
	};
	
	ctx.statements["for"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("variable name expected in FOR");
				
		ctx.expression = ctx.token;
		
		ctx.match('=');

		ctx.variables[ctx.expression] = ctx.parse_expression();
				
		ctx.match("to");

		ctx.variables["$limit"] = ctx.parse_expression();

		if (ctx.match("step"))
			ctx.variables["$step"] = ctx.parse_expression();
		else
			ctx.variables["$step"] = 1;
		
		ctx.buffer.clear();
		ctx.buf_to = "next";
	};
	
	ctx.statements["next"] = [](Context& ctx) {
		if (ctx.match_name() && ctx.token != ctx.expression)
			throw ParseError("loop variable mismatch in NEXT");
		
		const auto name = ctx.expression;
		const auto init = ctx.variables[name];
		const auto limit = ctx.variables["$limit"];
		const auto step = ctx.variables["$step"];
		const auto code = ctx.buffer;
		
		if (step > 0) {
			for (auto i = init; i <= limit; i += step) { 
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else if (step < 0) {
			for (auto i = init; i >= limit; i += step) {
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else {
			throw RunError("FOR loop with a STEP of zero");
		}
		
		ctx.variables.erase("$limit");
		ctx.variables.erase("$step");
	};

	ctx.statements["repeat"] = [](Context& ctx) {
		ctx.buffer.clear();
		ctx.buf_to = "until";
	};

	ctx.statements["until"] = [](Context& ctx) {
		const auto expr = ctx.line.substr(ctx.cursor);
		const auto code = ctx.buffer;
		do {
			for (const auto& i: code)
				ctx.eval(i);
			ctx.line = expr;
			ctx.cursor = 0;
		} while (!ctx.parse_expression());
	};
	
	ctx.statements["do"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("variable name expected in DO");
				
		ctx.expression = ctx.token;
		
		ctx.match('=');

		ctx.variables[ctx.expression] = ctx.parse_expression();
				
		ctx.match(',');

		ctx.variables["$limit"] = ctx.parse_expression();

		if (ctx.match(','))
			ctx.variables["$step"] = ctx.parse_expression();
		else
			ctx.variables["$step"] = 1;
		
		ctx.buffer.clear();
		ctx.buf_to = "done";
	};
	
	ctx.statements["done"] = [](Context& ctx) {
		const auto name = ctx.expression;
		const auto init = ctx.variables[name];
		const auto limit = ctx.variables["$limit"];
		const auto step = ctx.variables["$step"];
		const auto code = ctx.buffer;
		
		if (step > 0) {
			for (auto i = init; i <= limit; i += step) { 
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else if (step < 0) {
			for (auto i = init; i >= limit; i += step) {
				ctx.variables[name] = i;
				for (const auto& j: code)
					ctx.eval(j);
			}
		} else {
			throw RunError("DO loop with a step of zero");
		}
		
		ctx.variables.erase("$limit");
		ctx.variables.erase("$step");
	};
	
	ctx.statements["data"] = [](Context& ctx) {
		while (!ctx.at_end_of_statement()) {
			ctx.data.push_back(ctx.parse_expression());
			ctx.match(',');
		}
	};
	
	ctx.statements["read"] = [](Context& ctx) {
		while (ctx.match_name()) {
			if (ctx.pointer >= ctx.data.size())
				throw RunError("READ past the end of DATA");
			ctx.variables[ctx.token] = ctx.data[ctx.pointer];
			ctx.pointer++;
			ctx.match(',');
		}
	};
	
	ctx.statements["restore"] = [](Context& ctx) {
		std::size_t pos = 0;
		if (!ctx.at_end_of_statement())
			pos = ctx.parse_expression();
		if (pos >= ctx.data.size())
			throw RunError("RESTORE out of DATA range");
		else
			ctx.pointer = pos;
	};
	
	ctx.statements["randomize"] = [](Context& ctx) {
		if (ctx.at_end_of_statement())
			std::srand(std::time(nullptr));
		else
			std::srand(ctx.parse_expression());
	};
	
	ctx.statements["poke"] = [](Context& ctx) {
		std::size_t pos = ctx.parse_expression();
		if (pos >= ctx.data.size())
			throw RunError(
				"POKE out of DATA range");
		else if (!ctx.match(','))
			throw ParseError("data argument expected");
		else
			ctx.data[pos] = ctx.parse_expression();
	};
	
	ctx.statements["erase"] = [](Context& ctx) {
		while (ctx.match_name()) {
			ctx.variables.erase(ctx.token);
			ctx.arrays.erase(ctx.token);
			ctx.match(',');
		}
	};
	
	ctx.statements["option"] = [](Context& ctx) {
		if (ctx.match("explicit"))
			ctx.option_explicit = true;
		else if (ctx.match("base"))
			ctx.option_base = std::abs(ctx.parse_expression());
		else
			throw ParseError("option name expected");
	};
	
	ctx.statements["on"] = [](Context& ctx) {
		auto choice = ctx.parse_expression();
		
		ctx.match("gosub");
		
		std::vector<std::string> names;
		
		while (ctx.match_name()) {
			names.push_back(ctx.token);
			ctx.match(',');
		}
		
		if (1 <= choice && choice <= names.size()) {
			auto name = names[choice - 1];
			auto code = ctx.subroutines.at(name);
			for (const auto& i: code)
				ctx.eval(i); 
		}
	};
	
	ctx.statements["dim"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("array name expected in DIM");
				
		const auto name = ctx.token;
		
		if (ctx.arrays.count(name) > 0) {
			throw RunError("array already exists");
		} else if (ctx.match('[')) {
			const std::size_t size = ctx.parse_expression();
			if (ctx.match(']'))
				ctx.arrays[name].resize(size);
			else
				throw ParseError("unclosed bracket");
		} else {
			throw ParseError("bracket expected in DIM");
		}
	};
	
	ctx.statements["redim"] = [](Context& ctx) {
		ctx.match("preserve");
	
		if (!ctx.match_name())
			throw ParseError("array name expected in REDIM");
				
		const auto name = ctx.token;
		
		if (ctx.arrays.count(name) == 0) {
			throw RunError(std::string("array doesn't exist: ") + name);
		} else if (ctx.match('[')) {
			const std::size_t size = ctx.parse_expression();
			if (ctx.match(']'))
				ctx.arrays[name].resize(size);
			else
				throw ParseError("unclosed bracket");
		} else {
			throw ParseError("bracket expected in REDIM");
		}
	};

	ctx.statements["swap"] = [](Context& ctx) {
		if (!ctx.match_name())
			throw ParseError("variable name expected in SWAP");
		const auto v1 = ctx.token;
		
		ctx.match(',');
		if (!ctx.match_name())
			throw ParseError("variable name expected in SWAP");
		const auto v2 = ctx.token;
		
		if (ctx.option_explicit)
			std::swap(ctx.variables.at(v1), ctx.variables.at(v2));
		else
			std::swap(ctx.variables[v1], ctx.variables[v2]);
	};
	
	ctx.functions["abs"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::abs(args[0]);
		else
			throw ParseError("ABS expects exactly one argument");
	};
	
	ctx.functions["int"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::floor(args[0]);
		else
			throw ParseError("INT expects exactly one argument");
	};
	
	ctx.functions["sqr"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::sqrt(args[0]);
		else
			throw ParseError("SQR expects exactly one argument");
	};
	
	ctx.functions["sin"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::sin(args[0]);
		else
			throw ParseError("SIN expects exactly one argument");
	};
	
	ctx.functions["cos"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::cos(args[0]);
		else
			throw ParseError("COS expects exactly one argument");
	};
	
	ctx.functions["pi"] = [](Context& ctx, const Values& args) {
		return pi;
	};
	
	ctx.functions["e"] = [](Context& ctx, const Values& args) {
		return e;
	};
	
	ctx.functions["hypot"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return std::hypot(args[0], args[1]);
		else
			throw ParseError("HYPOT expects exactly two arguments");
	};

	ctx.functions["rad"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return args[0] * pi / 180;
		else
			throw ParseError("RAD expects exactly one argument");
	};

	ctx.functions["deg"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return args[0] * 180 / pi;
		else
			throw ParseError("DEG expects exactly one argument");
	};
	
	ctx.functions["min"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return std::fmin(args[0], args[1]);
		else
			throw ParseError("MIN expects exactly two arguments");
	};
	
	ctx.functions["max"] = [](Context& ctx, const Values& args) {
		if (args.size() == 2)
			return std::fmax(args[0], args[1]);
		else
			throw ParseError("MAX expects exactly two arguments");
	};
	
	ctx.functions["exp"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::exp(args[0]);
		else
			throw ParseError("EXP expects exactly one argument");
	};
	
	ctx.functions["log"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return std::log(args[0]);
		else
			throw ParseError("LOG expects exactly one argument");
	};
	
	ctx.functions["timer"] = [](Context& ctx, const Values& args) {
		return std::clock();
	};
	
	ctx.functions["rnd"] = [](Context& ctx, const Values& args) {
		return (double) std::rand() / ((double) RAND_MAX + 1); 
	};
	
	ctx.functions["peek"] = [](Context& ctx, const Values& args) {
		if (args.size() == 1)
			return ctx.data.at(args[0]);
		else
			throw ParseError("PEEK expects exactly one argument");
	};
	
	ctx.functions["sgn"] = [](Context& ctx, const Values& args) {
		if (args.size() != 1)
			throw ParseError("SGN expects exactly one argument");
		else if (args[0] < 0)
			return -1;
		else if (args[0] > 0)
			return 1;
		else
			return 0;
	};
}

}

#endif
