#include <iostream>
#include <fstream>

#include "batchbas.hpp"

const std::string revision_date = "2 October 2025";

const std::string help_text[] = {
	" [ -v | -h | -e CODE | -l FILE | PROGRAM ]",
	"",
	"Arguments:",
	"    PROGRAM:       run given program",
	"",
	"Options:",
	"    -h, --help:    show this message and quit",
	"    -v, --version: show version and quit",
	"    -e, --eval:    evaluate given CODE and quit",
	"    -l, --load:    load given FILE and go interactive",
	"    -q, --quiet:   suppress banner and prompt while interactive"
};

bool merge(const std::string& filename, basic::Context& ctx) {
	std::ifstream source(filename);
	if (source.is_open()) {
		std::string line;
		while(std::getline(source, line))
			if (line.length() > 0 && line[0] != '\'')
				ctx.eval(line);
		return true;
	} else {
		return false;
	}
}

bool try_to_merge(const std::string& filename, basic::Context& ctx) {
	try {
		return merge(filename, ctx);
	} catch (const basic::ParseError& e) {
		std::cerr << "Parsing error in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const basic::RunError& e) {
		std::cerr << "Runtime error in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const std::bad_function_call& e) {
		std::cerr << "Bad function call in line:\n"
			<< ctx.line << "\n" << e.what() << std::endl;
		return false;
	} catch (const std::out_of_range& e) {
		std::cerr << "Out of range in line:\n"
			<< ctx.line << "\n" << std::endl; // No need for details here.
		return false;
	}
}

void do_merge(basic::Context& ctx) {
	if (!ctx.match_string())
		throw basic::ParseError("filename expected");
	else if (!merge(ctx.token, ctx))
		throw basic::RunError("can't open file");
};

void do_list(basic::Context& ctx) {
	if (!ctx.match_name()) {
		for (const auto& i: ctx.subroutines) {
			std::cout << i.first << std::endl;
		}
	} else if (ctx.subroutines.count(ctx.token) > 0) {
		const auto& s = ctx.subroutines.at(ctx.token);
		for (const auto& i: s)
			std::cout << i << "\n";
		std::cout << std::flush;
	} else {
		throw basic::RunError("subroutine not found");
	}
}

void do_delete(basic::Context& ctx) {
	if (!ctx.match_name())
		throw basic::ParseError("subroutine name expected");
	else
		ctx.subroutines.erase(ctx.token);
}

void do_new(basic::Context& ctx) {
	ctx.variables.clear();
	ctx.data.clear();
	ctx.subroutines.clear();
}

void do_save(basic::Context& ctx) {
	if (!ctx.match_string())
		throw basic::ParseError("filename expected");
	std::ofstream dest(ctx.token);
	if (dest.is_open()) {
		for (std::size_t i = 0; i < ctx.data.size(); i++) {
			if (i % 10 == 0)
				dest << "data ";
			dest << ctx.data[i];
			if (i % 10 == 9)
				dest << '\n';
			else
				dest << ", ";
		}
		if (ctx.data.size() > 0)
			dest << std::endl;
		for (const auto& i: ctx.variables)
			dest << "let " << i.first
				<< " = " << i.second << '\n';
		if (ctx.variables.size() > 0)
			dest << std::endl;
		for (const auto& i: ctx.subroutines) {
			dest << "sub " << i.first << '\n';
			for (const auto& j: i.second)
				dest << j << '\n';
			dest << "return" << std::endl;
		}
	} else {
		throw basic::RunError("can't open file");
	}
}

int main(int argc, char* argv[]) {
	basic::Context ctx;
	basic::load_builtins(ctx);

	ctx.statements["merge"] = do_merge;
	ctx.statements["quit"] = [](basic::Context& ctx) { std::exit(0); };

	std::vector<std::string> args;
	args.reserve(argc);
	for (int i = 0; i < argc; i++)
		args.push_back(argv[i]);

	bool quiet = false;
	bool interactive = false;

	if (args.size() < 2) {
		interactive = true;
	} else if (args[1] == "-v" || args[1] == "--version") {
		std::cout << "Batch Basic " << basic::version_number << "\n";
		std::cout << "Revision " << revision_date << std::endl;
		return 0;
	} else if (args[1] == "-h" || args[1] == "--help") {
		std::cout << "Usage: " << args[0];
		for (const auto& i: help_text)
			std::cout << i << '\n';
		std::cout << std::flush;
		return 0;
	} else if (args[1] == "-e" || args[1] == "--eval") {
		if (args.size() > 2) {
			ctx.line = args[2];
			ctx.cursor = 0; 
			ctx.run_statements();
		} else {
			std::cerr << "Missing argument to "
				<< args[1] << std::endl;
			return 1;
		}
	} else if (args[1] == "-l" || args[1] == "--load") {
		if (args.size() < 3) {
			std::cerr << "Missing argument to "
				<< args[1] << std::endl;
			return 1;
		}
		if (try_to_merge(args[2], ctx))
			std::cout << "Loaded " << args[2] << std::endl;
		else
			std::cerr << "Failed to load "
				<< args[2] << std::endl;
		interactive = true;
	} else if (args[1] == "-q" || args[1] == "--quiet") {
		quiet = true;
		interactive = true;
	} else if (!try_to_merge(args[1], ctx)) {
		return 2;
	} else if (ctx.subroutines.count("main") > 0) {
		auto code = ctx.subroutines["main"];
		for (const auto& i: code)
			ctx.eval(i); 
	}
	
	if (!interactive) return 0;

	ctx.statements["clear"] = [](basic::Context& ctx) {
		ctx.variables.clear();
	};
	ctx.statements["load"] = [](basic::Context& ctx) {
		do_new(ctx);
		do_merge(ctx);
	};
	ctx.statements["new"] = do_new;
	ctx.statements["delete"] = do_delete;
	ctx.statements["list"] = do_list;
	
	ctx.statements["save"] = do_save;

	if (!quiet) {
		std::cout << "Batch Basic " << basic::version_number << '\n';
		std::cout << ctx.statements.size() << " statements and ";
		std::cout << ctx.functions.size() << " functions defined.\n";
		std::cout << "Enter code or QUIT to exit:\n";
		std::cout << "> " << std::flush;
	}

	std::string line;
	
	while(std::getline(std::cin, line)) {
		try {
			if (line.length() > 0 && line[0] != '\'')
				ctx.eval(line);
			if (ctx.buf_to == "" && !quiet)
				std::cout << "OK\n";
		} catch (const basic::ParseError& e) {
			std::cerr << "Parsing error in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const basic::RunError& e) {
			std::cerr << "Runtime error in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const std::bad_function_call& e) {
			std::cerr << "Bad function call in line:\n"
				<< line << "\n" << e.what() << std::endl;
		} catch (const std::out_of_range& e) {
			std::cerr << "Out of range in line:\n"
				<< line << "\n" << std::endl; // No need for details here.
		}
		if (!quiet)
			std::cout << (ctx.buf_to == "" ? "> " : "... ") << std::flush;
	}
	
	return 0;
}
