#!/usr/bin/env python3

"""BASIC-like expression calculator and operating system shell."""

from __future__ import division
from __future__ import print_function

import re
import math

import os
import os.path
import subprocess

statement_names = [
	'print', 'let', 'def', 'load', 'cd',
	'if', 'while', 'data', 'read', 'restore',
	'mkdir', 'rmdir', 'copy', 'name', 'delete']

# Each RE must have exactly one parenthesized group, even if not needed.
re_stmt = re.compile('\s*(' + '|'.join(statement_names) + ')\\b', re.I)
re_fn = re.compile('\s*(fn)', re.I) # Allow for DEF FNa$ = ...
re_then = re.compile('\s*(then)\\b', re.I)
re_else = re.compile('\s*(else)\\b', re.I)
re_wend = re.compile('\s*(wend)\\b', re.I)
re_name = re.compile('\s*([a-z][a-z0-9]*\$?)', re.I)
re_float = re.compile('\s*(-?\d+(\.\d*)?)')
re_string = re.compile('\s*"([^"]*)"')
re_rem = re.compile('\s*\'(.*)$')
re_eol = re.compile('(\s*)$')
re_assign = re.compile('\s*(=)')
re_specop = re.compile('\s*(iif|min|max|shell|path)\\b', re.I)
re_or = re.compile('\s*(or)\\b', re.I)
re_and = re.compile('\s*(and)\\b', re.I)
re_not = re.compile('\s*(not)\\b', re.I)
re_cmp = re.compile('\s*(<>|=|<=|<|>=|>)')
re_addop = re.compile('\s*([+-])')
re_mulop = re.compile('\s*([*/\\\\])')
re_powop = re.compile('\s*(^)')
re_lparen = re.compile('\s*(\()')
re_rparen = re.compile('\s*(\))')
re_comma = re.compile('\s*(,)')
re_semi = re.compile('\s*(;)')
re_colon = re.compile('\s*(:)')

class Parser:
	def __init__(self, text):
		self.text = text
		self.pos = 0
		self.token = None

	def match(self, re):
		m = re.match(self.text, self.pos)
		if m:
			self.token = m.group(1)
			self.pos = m.end()
			return True
		else:
			return False

	def peek(self, re):
		return re.match(self.text, self.pos) != None

def parse_line(parser):
	stlist = parse_stmtlist(parser)

	if parser.match(re_rem) or parser.match(re_eol):
		return stlist
	else:
		raise SyntaxError("statement or comment expected")

def parse_stmtlist(parser):
	stlist = []

	if parser.match(re_stmt):
		stlist.append(parse_statement(parser))
		go_on = True
	elif parser.peek(re_name):
		stlist.append(parse_let(parser))
		go_on = True
	else:
		go_on = False
	
	if go_on:
		while parser.match(re_colon):
			if parser.match(re_stmt):
				stlist.append(parse_statement(parser))
			elif parser.peek(re_name):
				stlist.append(parse_let(parser))
			else:
				break
	return stlist

def parse_statement(parser):
	stmt = parser.token.lower()
	if stmt == "print":
		return parse_print(parser)
	elif stmt == "let":
		return parse_let(parser)
	elif stmt == "def":
		return parse_def(parser)
	elif stmt == "load":
		return parse_load(parser)
	elif stmt == "cd":
		return parse_cd(parser)
	elif stmt == "if":
		return parse_if(parser)
	elif stmt == "while":
		return parse_while(parser)
	elif stmt == "data":
		return parse_data(parser)
	elif stmt == "read":
		return parse_read(parser)
	elif stmt == "restore":
		return parse_restore(parser)
	elif stmt in ["mkdir", "rmdir", "delete"]:
		return parse_1str(parser, stmt)
	elif stmt in ["copy", "name"]:
		return parse_2str(parser, stmt)
	else:
		raise SyntaxError("not a statement: " + parser.token)

def parse_print(parser):
	if at_stmt_end(parser):
		return ("print", [])

	exprs = [parse_expression(parser)]
	while parser.match(re_comma) or parser.match(re_semi):
		if at_stmt_end(parser):
			return ("print", exprs)

		if parser.token == ",":
			exprs.append(("string", "literal", "\t\t"))
		exprs.append(parse_expression(parser))

	exprs.append(("string", "literal", "\n"))
	return ("print", exprs)

def at_stmt_end(parser):
	return (parser.peek(re_colon)
		or parser.peek(re_rem)
			or parser.peek(re_eol))

def parse_let(parser):
	if not parser.match(re_name):
		raise SyntaxError("variable name expected")
		
	var = name2ref(parser.token.lower())
	
	if not parser.match(re_assign):
		raise SyntaxError("assignment expected")
	
	expr = parse_expression(parser)
	
	if var[0] == expr[0]:
		return ("let", var, expr)
	else:
		raise TypeError("type mismatch in assignment")

def name2ref(name):
	if name[-1] == "$":
		return ("string", "variable", name)
	else:
		return ("float", "variable", name)

def parse_def(parser):
	parser.match(re_fn) # Don't mandate the FN keyword.

	if not parser.match(re_name):
		raise SyntaxError("variable name expected")

	name = name2ref(parser.token.lower())
	
	arglist = []
	if parser.match(re_lparen):
		if not parser.match(re_rparen):
			if not parser.match(re_name):
				raise SyntaxError("variable name expected")
			arglist.append(parser.token.lower())
			while parser.match(re_comma):
				if not parser.match(re_name):
					raise SyntaxError("varname expected")
				arglist.append(parser.token.lower())
			if not parser.match(re_rparen):
				raise SyntaxError("unclosed parenthesis")
	
	if not parser.match(re_assign):
		raise SyntaxError("assignment expected")

	expr = parse_expression(parser)

	if expr[0] == name[0]:
		return ("def", name[2], arglist, expr)
	else:
		raise TypeError("function name must match return type")

def parse_load(parser):
	expr = parse_expression(parser)
	
	if expr[0] == "string":
		return ("load", expr)
	else:
		raise TypeError("string expression expected")

def parse_cd(parser):
	if at_stmt_end(parser):
		return ("cd", None)
	else:
		dirname = parse_expression(parser)
		if dirname[0] == "string":
			return ("cd", dirname)
		else:
			raise TypeError("string expression expected")

def parse_if(parser):
	cond = parse_expression(parser)
	
	if cond[0] != "float":
		raise TypeError("float expression expected")
		
	if not parser.match(re_then):
		raise SyntaxError("IF without THEN")

	true_branch = parse_stmtlist(parser)
	
	if parser.match(re_else):
		false_branch = parse_stmtlist(parser)
	else:
		false_branch = []
	
	return ("if", cond, true_branch, false_branch)

def parse_while(parser):
	cond = parse_expression(parser)
	
	if cond[0] != "float":
		raise TypeError("float expression expected")
	
	if not parser.match(re_colon):
		raise SyntaxError(": expected after WHILE condition")
	
	stlist = parse_stmtlist(parser)
	parser.match(re_wend) # By default goes to the end of the line.
	return ("while", cond, stlist)
	
def parse_data(parser):
	if parser.match(re_float):
		data = [("float", "literal", float(parser.token))]
	elif parser.match(re_string):
		data = [("string", "literal", parser.token)]
	else:
		raise SyntaxError("literal expected")
	
	while parser.match(re_comma):
		if parser.match(re_float):
			data.append(("float", "literal", float(parser.token)))
		elif parser.match(re_string):
			data.append(("string", "literal", parser.token))
		else:
			raise SyntaxError("literal expected")

	return ("data", data)

def parse_read(parser):
	if parser.match(re_name):
		names = [name2ref(parser.token.lower())]
	else:
		raise SyntaxError("variable name expected")

	while parser.match(re_comma):
		if parser.match(re_name):
			names.append(name2ref(parser.token.lower()))
		else:
			raise SyntaxError("variable name expected")
	
	return ("read", names)

def parse_restore(parser):
	if at_stmt_end(parser):
		return ("restore", ("float", "literal", 0))

	pos = parse_expression(parser)
	
	if pos[0] == "float":
		return ("restore", pos)
	else:
		raise TypeError("float expression expected")

def parse_1str(parser, statement):
	expr = parse_expression(parser)
	
	if expr[0] == "string":
		return (statement, expr)
	else:
		raise TypeError("string expression expected")

def parse_2str(parser, statement):
	source = parse_expression(parser)
	
	if source[0] != "string":
		raise TypeError("string expression expected")
	elif not parser.match(re_comma):
		raise SyntaxError("comma expected after first argument")

	target = parse_expression(parser)

	if target[0] == "string":
		return (statement, source, target)
	else:
		raise TypeError("string expression expected")

def parse_expression(parser):
	if parser.match(re_specop):
		op = parser.token.lower()
		if op == "iif":
			return parse_iif(parser)
		elif op == "min":
			return parse_min(parser)
		elif op == "max":
			return parse_max(parser)
		elif op == "shell":
			return parse_shell(parser)
		elif op == "path":
			return parse_path(parser)
	else:
		return parse_disjunction(parser)

def parse_iif(parser):
	if not parser.match(re_lparen):
		raise SyntaxError("parenthesis expected")

	cond = parse_expression(parser)
	
	if cond[0] != "float":
		raise TypeError("condition expected in iif")
	
	if not parser.match(re_comma):
		raise SyntaxError("comma expected after iif condition")
	
	ift = parse_expression(parser)
	
	if not parser.match(re_comma):
		raise SyntaxError("comma expected after iif true branch")
	
	iff = parse_expression(parser)

	if not parser.match(re_rparen):
		raise SyntaxError("unclosed parenthesis")
	elif ift[0] != iff[0]:
		raise TypeError("branches must have the same type in iif")
	
	return (ift[0], "iif", cond, ift, iff)

def parse_min(parser):
	if not parser.match(re_lparen):
		raise SyntaxError("parenthesis expected")

	expr = parse_exprlist(parser)

	if not parser.match(re_rparen):
		raise SyntaxError("unclosed parenthesis")
	
	for i in range(1, len(expr)):
		if expr[i][0] != expr[0][0]:
			raise TypeError("argument type mismatch")
	
	return (expr[0][0], "min", expr)

def parse_max(parser):
	if not parser.match(re_lparen):
		raise SyntaxError("parenthesis expected")

	expr = parse_exprlist(parser)

	if not parser.match(re_rparen):
		raise SyntaxError("unclosed parenthesis")
	
	for i in range(1, len(expr)):
		if expr[i][0] != expr[0][0]:
			raise TypeError("argument type mismatch")
	
	return (expr[0][0], "max", expr)

def parse_shell(parser):
	if not parser.match(re_lparen):
		raise SyntaxError("parenthesis expected")

	expr = parse_exprlist(parser)

	if not parser.match(re_rparen):
		raise SyntaxError("unclosed parenthesis")
	
	return ("string", "shell", expr)

def parse_path(parser):
	if not parser.match(re_lparen):
		raise SyntaxError("parenthesis expected")

	expr = parse_exprlist(parser)

	if not parser.match(re_rparen):
		raise SyntaxError("unclosed parenthesis")
	
	return ("string", "path", expr)

def parse_disjunction(parser):
	lside = parse_conjunction(parser)
	while parser.match(re_or):
		rside = parse_conjunction(parser)
		if lside[0] != "float" or rside[0] != "float":
			raise TypeError("boolean value expected")
		else:
			lside = ("float", "or", lside, rside)
	return lside

def parse_conjunction(parser):
	lside = parse_negation(parser)
	while parser.match(re_and):
		rside = parse_negation(parser)
		if lside[0] != "float" or rside[0] != "float":
			raise TypeError("boolean value expected")
		else:
			lside = ("float", "and", lside, rside)
	return lside

def parse_negation(parser):
	if parser.match(re_not):
		comp = parse_comparison(parser)
		if comp[0] == "string":
			raise TypeError("can't negate a string")
		return ("float", "not", comp)
	else:
		return parse_comparison(parser)

def parse_comparison(parser):
	lside = parse_arithmetic(parser)
	
	if parser.match(re_cmp):
		op = parser.token
		rside = parse_arithmetic(parser)
		if lside[0] != rside[0]:
			raise TypeError("can't compare string to float")
		return ("float", op, lside, rside)
	else:
		return lside

def parse_arithmetic(parser):
	term1 = parse_term(parser)
	
	while parser.match(re_addop):
		op = parser.token
		term2 = parse_term(parser)
		if op == "+":
			if term1[0] != term2[0]:
				raise TypeError("can't add string to float")
		elif term1[0] == "string" or term2[0] == "string":
			raise TypeError("can't subtract (from) string")
		term1 = (term1[0], op, term1, term2)
	
	return term1

def parse_term(parser):
	factor1 = parse_factor(parser)
	
	while parser.match(re_mulop):
		op = parser.token
		if factor1[0] == "string":
			raise TypeError("can't multiply or divide strings")
		factor2 = parse_factor(parser)
		if factor2[0] == "string":
			raise TypeError("can't multiply or divide strings")
		factor1 = ("float", op, factor1, factor2)
	
	return factor1

def parse_factor(parser):
	val1 = parse_value(parser)
	if parser.match(re_powop):
		if val1[0] == "string":
			raise TypeError("can't raise string to power")
		val2 = parse_factor(parser)
		if val2[0] == "string":
			raise TypeError("can't raise string to power")
		return ("float", "^", val1, val2)
	else:
		return val1

def parse_value(parser):
	if not parser.match(re_addop):
		signum = None
	elif parser.token == "-":
		signum = -1
	else:
		signum = 1
	
	if parser.match(re_float):
		val = ("float", "literal", float(parser.token))
		if signum == -1:
			return ("float", "minus", val)
		else:
			return val
	elif parser.match(re_string):
		if signum != None:
			raise TypeError("strings have no signum")
		else:
			return ("string", "literal", str(parser.token))
	elif parser.match(re_lparen):
		expr = parse_expression(parser)
		if not parser.match(re_rparen):
			raise SyntaxError("unclosed parenthesis")
		elif signum == None:
			return expr
		elif expr[0] == "string":
			raise TypeError("strings have no signum")
		elif signum == -1:
			return ("float", "minus", expr)
		else:
			return expr
	elif parser.match(re_name):
		ref = name2ref(parser.token.lower())
		if not parser.match(re_lparen):
			return ref
		else:
			if parser.match(re_rparen):
				args = []
			else:
				args = parse_exprlist(parser)
				if not parser.match(re_rparen):
					raise SyntaxError("unclosed parens")
			expr = (ref[0], "funcall", ref[2], args)
			if signum == None:
				return expr
			elif expr[0] == "string":
				raise TypeError("strings have no signum")
			elif signum == -1:
				return ("float", "minus", expr)
			else:
				return expr
	else:
		raise SyntaxError("value expected")

def parse_exprlist(parser):
	expr = [parse_expression(parser)]
	while parser.match(re_comma):
		expr.append(parse_expression(parser))
	return expr


class Scope:
	def __init__(self, parent = None):
		self.names = {}
		self.parent = parent
	
	def __getitem__(self, key):
		if key in self.names:
			return self.names[key]
		elif self.parent != None:
			return self.parent[key]
		else:
			raise KeyError("undefined variable: " + str(key))
	
	def __setitem__(self, key, value):
		if self.parent != None:
			self.parent[key] = value
		else:
			self.names[key] = value
	
	def __contains__(self, key):
		if key in self.names:
			return True
		elif self.parent != None:
			return key in self.parent
		else:
			return False

def eval_line(ast_list, scope, functions):
	for ast in ast_list:
		if ast[0] == "print":
			eval_print(ast[1], scope, functions)
		elif ast[0] == "let":
			eval_let(ast[1], ast[2], scope, functions)
		elif ast[0] == "def":
			eval_def(ast[1], ast[2], ast[3], scope, functions)
		elif ast[0] == "load":
			filename = eval_expr(ast[1], scope, functions)
			with open(filename, "r") as handle:
				load_file(handle, scope, functions)
		elif ast[0] == "cd":
			eval_cd(ast[1], scope, functions)
		elif ast[0] == "if":
			if eval_expr(ast[1], scope, functions):
				eval_line(ast[2], scope, functions)
			else:
				eval_line(ast[3], scope, functions)
		elif ast[0] == "while":
			while eval_expr(ast[1], scope, functions):
				eval_line(ast[2], scope, functions)
		elif ast[0] == "data":
			for i in ast[1]:
				data_store.append(i)
		elif ast[0] == "read":
			eval_read(ast[1], scope, functions)
		elif ast[0] == "restore":
			eval_restore(ast[1], scope, functions)
		elif ast[0] == "mkdir":
			os.mkdir(eval_expr(ast[1], scope, functions))
		elif ast[0] == "rmdir":
			os.rmdir(eval_expr(ast[1], scope, functions))
		elif ast[0] == "copy":
			raise RuntimeError("not implemented")
		elif ast[0] == "name":
			os.rename(
				eval_expr(ast[1], scope, functions),
				eval_expr(ast[2], scope, functions))
		elif ast[0] == "delete":
			os.remove(eval_expr(ast[1], scope, functions))
		else:
			raise ValueError("not a statement: " + str(ast[0]))

def eval_print(expr_list, scope, functions):
	for expr in expr_list:
		if expr[0] == "float":
			value = eval_expr(expr, scope, functions)
			print("{0:g}".format(value), end='')
		else:
			print(eval_expr(expr, scope, functions), end='')

def eval_let(var, expr, scope, functions):
	scope[var[2]] = eval_expr(expr, scope, functions)

def eval_def(name, argnames, expr, scope, functions):
	functions[name] = ([name2ref(n) for n in argnames], expr)

def load_file(handle, scope, functions):
	line_num = 0
	for i in handle:
		line_num += 1
		parser = Parser(i)
		try:
			eval_line(parse_line(parser), scope, functions)
		except Exception as e:
			print("In {0}, line {1}, col {2}:".format(
				handle.name, line_num, parser.pos))
			print(e)
			break

def read_file(filename):
	with open(filename, "r") as f:
		return f.read()

def eval_cd(dirname, scope, functions):
	if dirname != None:
		dirname = eval_expr(dirname, scope, functions)
	else:
		dirname = "~"
	os.chdir(os.path.expanduser(dirname))

def eval_read(varnames, scope, functions):
	global data_cursor
	for i in varnames:
		if data_cursor >= len(data_store):
			raise IndexError(
				"READ past the end of DATA")
		elif i[0] != data_store[data_cursor][0]:
			raise RuntimeError("DATA type mismatch")
		else:
			scope[i[2]] = data_store[data_cursor][2]
			data_cursor += 1

def eval_restore(cursor, scope, functions):
	global data_cursor
	cursor = eval_expr(cursor, scope, functions)
	if cursor < 0:
		cursor = len(data_store - cursor)
	if cursor < 0:
		raise IndexError("RESTORE past the start of DATA")
	elif cursor >= len(data_store):
		raise IndexError("RESTORE past the end of DATA")
	else:
		data_cursor = cursor

def eval_expr(expr, scope, functions):
	if expr[1] == "literal":
		return expr[2]
	elif expr[1] == "variable":
		return scope[expr[2]]
	elif expr[1] == "funcall":
		return eval_funcall(expr[2], expr[3], scope, functions)
	elif expr[1] == "builtin":
		return expr[2](scope, functions)
	elif expr[1] == "minus":
		return -eval_expr(expr[2], scope, functions)
	elif expr[1] == "^":
		return (eval_expr(expr[2], scope, functions)
			** eval_expr(expr[3], scope, functions))
	elif expr[1] == "*":
		return (eval_expr(expr[2], scope, functions)
			* eval_expr(expr[3], scope, functions))
	elif expr[1] == "/":
		return (eval_expr(expr[2], scope, functions)
			/ eval_expr(expr[3], scope, functions))
	elif expr[1] == "\\":
		return (eval_expr(expr[2], scope, functions)
			// eval_expr(expr[3], scope, functions))
	elif expr[1] == "+":
		return (eval_expr(expr[2], scope, functions)
			+ eval_expr(expr[3], scope, functions))
	elif expr[1] == "-":
		return (eval_expr(expr[2], scope, functions)
			- eval_expr(expr[3], scope, functions))
	elif expr[1] == "<=":
		return -(eval_expr(expr[2], scope, functions)
			<= eval_expr(expr[3], scope, functions))
	elif expr[1] == "<":
		return -(eval_expr(expr[2], scope, functions)
			< eval_expr(expr[3], scope, functions))
	elif expr[1] == "=":
		return -(eval_expr(expr[2], scope, functions)
			== eval_expr(expr[3], scope, functions))
	elif expr[1] == "<>":
		return -(eval_expr(expr[2], scope, functions)
			!= eval_expr(expr[3], scope, functions))
	elif expr[1] == ">":
		return -(eval_expr(expr[2], scope, functions)
			> eval_expr(expr[3], scope, functions))
	elif expr[1] == ">=":
		return -(eval_expr(expr[2], scope, functions)
			>= eval_expr(expr[3], scope, functions))
	elif expr[1] == "not":
		if not eval_expr(expr[2], scope, functions):
			return -1
		else:
			return 0
	elif expr[1] == "and":
		if (eval_expr(expr[2], scope, functions)
				and eval_expr(expr[3], scope, functions)):
			return -1
		else:
			return 0
	elif expr[1] == "or":
		if (eval_expr(expr[2], scope, functions)
				or eval_expr(expr[3], scope, functions)):
			return -1
		else:
			return 0
	elif expr[1] == "iif":
		if eval_expr(expr[2], scope, functions):
			return eval_expr(expr[3], scope, functions)
		else:
			return eval_expr(expr[4], scope, functions)
	elif expr[1] == "min":
		m = eval_expr(expr[2][0], scope, functions)
		for i in range(1, len(expr[2])):
			n = eval_expr(expr[2][i], scope, functions)
			if n < m:
				m = n
		return m
	elif expr[1] == "max":
		m = eval_expr(expr[2][0], scope, functions)
		for i in range(1, len(expr[2])):
			n = eval_expr(expr[2][i], scope, functions)
			if n > m:
				m = n
		return m
	elif expr[1] == "shell":
		cmd = [eval_expr(i, scope, functions) for i in expr[2]]
		try:
			output = subprocess.check_output(cmd).decode()
			scope["err"] = 0
			scope["err$"] = ""
		except subprocess.CalledProcessError as e:
			output = ""
			scope["err"] = e.returncode
			scope["err$"] = str(e)
		return output
	elif expr[1] == "path":
		args = [eval_expr(i, scope, functions) for i in expr[2]]
		return os.path.join(*args)
	else:
		raise RuntimeError("not implemented: " + expr[1])

def eval_funcall(name, args, scope, functions):
	fn = functions.get(name)
	if fn == None:
		fn = builtins.get(name)
	
	if fn == None:
		raise RuntimeError("no such function: " + name)
	elif len(fn[0]) > len(args):
		raise RuntimeError("not enough arguments to " + name)
	elif len(fn[0]) < len(args):
		raise RuntimeError("too many arguments to " + name)

	fna = fn[0]
	fn_scope = Scope(scope)

	for i in range(len(fna)):
		if fna[i][0] != args[i][0]:
			raise RuntimeError("argument type mismatch")
		else:
			fn_scope.names[fna[i][2]] = eval_expr(
				args[i], scope, functions)
	
	return eval_expr(fn[1], fn_scope, functions)

def make_builtin(t, argnames, fn):
	return ([name2ref(n) for n in argnames], (t, "builtin", fn))

builtins = {
	"pi": make_builtin("float", [],
		lambda s, fn: math.pi),
	"int": make_builtin("float", ["n"],
		lambda s, fn: math.trunc(s["n"])),
	"abs": make_builtin("float", ["n"],
		lambda s, fn: abs(s["n"])),
	"sqr": make_builtin("float", ["n"],
		lambda s, fn: math.sqrt(s["n"])),
	"sin": make_builtin("float", ["n"],
		lambda s, fn: math.sin(s["n"])),
	"cos": make_builtin("float", ["n"],
		lambda s, fn: math.cos(s["n"])),
	"rad": make_builtin("float", ["n"],
		lambda s, fn: math.radians(s["n"])),
	"deg": make_builtin("float", ["n"],
		lambda s, fn: math.degrees(s["n"])),
	"mod": make_builtin("float", ["a", "b"],
		lambda s, fn: s["a"] % s["b"]),
	"hypot": make_builtin("float", ["a", "b"],
		lambda s, fn: math.sqrt(s["a"] ** 2 + s["b"] ** 2)),
	"e": make_builtin("float", [],
		lambda s, fn: math.e),
	"exp": make_builtin("float", ["n"],
		lambda s, fn: math.exp(s["n"])),
	"exp10": make_builtin("float", ["n"],
		lambda s, fn: math.exp10(s["n"])),
	"log": make_builtin("float", ["n"],
		lambda s, fn: math.log(s["n"])),
	"log10": make_builtin("float", ["n"],
		lambda s, fn: math.log10(s["n"])),
	
	"len": make_builtin("float", ["s$"],
		lambda s, fn: len(s["s$"])),
	"str$": make_builtin("string", ["n"],
		lambda s, fn: "{0:g}".format(s["n"])),
	"val": make_builtin("float", ["s$"],
		lambda s, fn: float(s["s$"])),
	"chr$": make_builtin("string", ["n"],
		lambda s, fn: chr(s["n"])),
	"asc": make_builtin("float", ["s$"],
		lambda s, fn: ord(s["s$"])),
	"bin$": make_builtin("string", ["n"],
		lambda s, fn: bin(s["n"])),
	"hex$": make_builtin("string", ["n"],
		lambda s, fn: hex(s["n"])),
	"oct$": make_builtin("string", ["n"],
		lambda s, fn: oct(s["n"])),

	"string$": make_builtin("string", ["n", "s$"],
		lambda s, fn: s["s$"] * int(s["n"])),
	"space$": make_builtin("string", ["n"],
		lambda s, fn: " " * int(s["n"])),

	"left$": make_builtin("string", ["s$", "n"],
		lambda s, fn: s["s$"][ :int(s["n"]) ]),
	# String indices in Basic start from 1.
	"mid$": make_builtin("string", ["s$", "f", "n"],
		lambda s, fn: s["s$"][ int(s["f"])-1 : int(s["f"]-1+s["n"]) ]),
	"right$": make_builtin("string", ["s$", "n"],
		lambda s, fn: s["s$"][ -int(s["n"]): ]),
	"lcase$": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].lower()),
	"ucase$": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].upper()),
	"ltrim$": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].lstrip()),
	"trim$": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].strip()),
	"rtrim$": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].rstrip()),
	# String indices in Basic start from 1,
	# so find() returns 0 on failure, and can be used as boolean.
	"find": make_builtin("float", ["s1$", "s2$"],
		lambda s, fn: s["s1$"].find(s["s2$"]) + 1),
	"rfind": make_builtin("float", ["s1$", "s2$"],
		lambda s, fn: s["s1$"].rfind(s["s2$"]) + 1),
	"replace$": make_builtin("string", ["s1$", "s2$", "s3$"],
		lambda s, fn: s["s1$"].replace(s["s2$"], s["s3$"])),
	"isdigit": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].isdigit()),
	"isalpha": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].isalpha()),
	"isalnum": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].isalnum()),
	"isspace": make_builtin("string", ["s$"],
		lambda s, fn: s["s$"].isspace()),

	"input": make_builtin("string", ["s$"],
		lambda s, fn: float(input(s["s$"]))),
	"input$": make_builtin("string", ["s$"],
		lambda s, fn: input(s["s$"])),
	
	"environ$": make_builtin("string", ["v$"],
		lambda s, fn: os.environ[ s["v$"] ]),
	"workdir$": make_builtin("string", [],
		lambda s, fn: os.getcwd()),
	"file$": make_builtin("string", ["f$"],
		lambda s, fn: read_file(s["f$"])),

	"fileexists": make_builtin("float", ["f$"],
		lambda s, fn: os.path.exists(s["f$"])),
	"filelen": make_builtin("float", ["f$"],
		lambda s, fn: os.path.getsize(s["f$"])),
	"isfile": make_builtin("float", ["f$"],
		lambda s, fn: os.path.isfile(s["f$"])),
	"isdir": make_builtin("float", ["f$"],
		lambda s, fn: os.path.isdir(s["f$"])),
	"abspath$": make_builtin("string", ["f$"],
		lambda s, fn: os.path.abspath(s["f$"])),
}

data_store = []
data_cursor = 0

def repl(banner, scope, functions):
	global data_cursor
	
	print(banner)

	while True:
		prompt = "prompt$" in scope and scope["prompt$"] or ">"
		try:
			line = input(prompt + " ")
		except EOFError:
			break
		except KeyboardInterrupt:
			break
		
		line = line.lstrip()
		if line.lower() == "bye":
			print("See you around!")
			break
		elif line.lower() == "cls":
			print("\x1b[2J\x1b[H", end='')
			continue
		elif line.lower() == "dir":
			for i in sorted(os.listdir(".")):
				if os.path.isdir(i):
					print("{0:<10s} {1}".format(
						"<DIR>", i))
				else:
					print("{0:>10d} {1}".format(
						os.path.getsize(i), i))
			continue
		elif line.lower() == "pwd":
			print(os.getcwd())
			continue
		elif line.lower() == "clear":
			scope.names.clear()
			continue
		elif line.lower() == "new":
			data_store.clear()
			data_cursor = 0
			functions.clear()
			continue
		elif line.lower().startswith("help"):
			topic = line[4:].strip()
			if topic in help_text:
				print(help_text[topic])
			else:
				print("No help on " + topic + ".")
			continue
		elif line[0] == "?":
			try:
				parser = Parser(line[1:])
				ast = [parse_print(parser)]
				eval_line(ast, scope, functions)
			except Exception as e:
				print("Column {0}: {1}.".format(parser.pos, e))
			continue
		elif line[0] == "!":
			try:
				subprocess.check_call(line[1:], shell=True)
				scope["err"] = 0
				scope["err$"] = ""
			except subprocess.CalledProcessError as e:
				scope["err"] = e.returncode
				scope["err$"] = str(e)
			continue
		
		parser = Parser(line)
		try:
			eval_line(parse_line(parser), scope, functions)
		except Exception as e:
			print("Column {0}: {1}.".format(parser.pos, e))

help_text = {}
help_text[""] = """
Usage example:

	i = 1: s = 0: while i <= 10: s = s + i: i = i + 1 wend: print s

	def fn factorial(n) = iif(n <= 0, 1, n * factorial(n - 1))
	n = 2 + 3
	?factorial(n)

Type "help <topic>" (without the quotes) to learn more about:

	statements	operators	functions
	commands	syntax		variables
"""

help_text["statements"] = """
Batch Basic supports the following statements:

PRINT [expression, ...]
	Display given expressions; separators determine spacing: semicolon
	for none, comma for tabs. End with a comma or semicolon to suppress
	the final newline.
[LET] variable = expression
	Assign the value of an expression to a variable. Expression type must
	match variable suffix ($ for strings, none for numbers).
DEF FN name([arguments...]) = expression
	Define a named function; expression type must match name suffix, as
	for variables. Argument types are checked at function call time.
LOAD string-expression
	Load and run the file denoted by the expression line by line, as if
	entered from the keyboard. Any error will interrupt the loading.
CD [string-expression]
	Change the current working directory to the one given, or else the
	user's home directory if entered by itself.

IF condition THEN statements... [ELSE statements...]
	Conditional statement, restricted to a single line. Don't place a
	colon in front of ELSE, if the clause is present.
WHILE condition : statements [WEND]
	Simple conditional loop, restricted to a single line; you can use
	the WEND keyword to terminate it early (otherwise it includes all
	remaining statements). Don't place a colon in front of WEND if
	using the keyword.
DATA literal [, literal...]
	Store one or more data items in internal memory; only string and
	float literals are allowed (no expressions).
READ variable [, variable...]
	Read one or more data items from internal memory into the given
	variables, advancing an internal index every time. Variable types
	must match the respective data items.
RESTORE [float-expression]
	Reset the internal data index to a previous position (zero by
	default). Negative values are counted from the end.

MKDIR string-expression
	Create a directory with the given name.
RMDIR string-expression
	Delete the directory with the given name, if it's empty.
COPY string-expression, string-expression
	Make a copy of the file given by the first argument.
	Not yet implemented.
NAME string-expression, string-expression
	Rename the file or directory given by the first argument. Second
	argument specifies the destination.
DELETE string-expression
	Delete the file with the given name.
"""

help_text["operators"] = """
Batch Basic has three categories of operators:

- binary arithmetic: +, -, *, /, \ (integer division), ^ (exponentiation);
- comparisons: <=, <, =, <>, >, >=;
- logic: AND, OR, NOT.

"+" can also be used to concatenate strings, and "-" to invert the sign of
an expression. Parentheses work as per the usual rules.

Additionally, there are five special operators: IIF, MIN, MAX, PATH and SHELL.
See "help special" (without the quotes) for how they work.
"""

help_text["functions"] = """
Batch Basic functions can be divided as follows:

Math functions:
	PI(), INT(n), ABS(n), SQR(n), MOD(a, b),
	SIN(n), COS(n), RAD(n), DEG(n), HYPOT(a, b),
	E(), EXP(n), EXP10(n), LOG(n), LOG10(n).
String functions:
	LEFT$(s$, n), MID$(s$, n, len), RIGHT$(s$, n),
	LCASE$(s$), UCASE$(s$),	LTRIM$(s$), TRIM$(s$), RTRIM$(s$),
	FIND(in$, sub$), RFIND(in$, sub$), REPLACE$(s$, old$, new$),
	ISDIGIT(s$), ISALPHA(s$), ISALNUM(s$), ISSPACE(s$).
Conversion functions:
	LEN(s$), STR$(n), VAL(s$), CHR$(n), ASC(s$),
	BIN$(s$), HEX$(s$), OCT$(s$), STRING$(n, s$), SPACE$(n).
Input functions:
	INPUT(prompt$), INPUT$(prompt$).
Operating system functions:
	WORKDIR$(), ENVIRON$(varname$), FILE$(filename$),
	FILEEXISTS(f$), FILELEN(f$), ISFILE(f$), ISDIR(f$), ABSPATH$(f$).
"""

help_text["commands"] = """
Batch Basic comes with a number of additional commands for convenience:

HELP [topic-name]
	Shows help on the given topic, if available.
BYE	quits the interpreter and returns to the operating system. You can
	also just interrupt the program with Ctrl-D or Ctrl-C.
CLS	clears the screen; this only works on terminal emulators that
	understand ANSI escape sequences.
CLEAR	deletes all global variables.
NEW	deletes all data from internal memory, and all user-defined functions.
PWD	shows the current working directory.
DIR	shows a list of entries in the current working directory.
?	is a shortcut for PRINT, taking an expression list in the same format.
!	passes the rest of the line to the operating system shell, and
	displays any resulting output; the exit code will be placed in a
	variable called ERR, and any error message if the call fails will be
	in ERR$.

Commands are only available in interactive mode, not while loading a file,
and must appear on a line by themselves, unlike statements.
"""

help_text["syntax"] = """
Batch Basic is a line-oriented language. No statement can be written across
multiple lines, so each line can be parsed and interpreted as soon as entered.

A line is composed of zero or more statements separated by a colon character,
and optionally ends with a comment. Comments start with a single-quote
character and last to the end of the line.

Batch Basic is case insensitive. Statements and other keywords, along with
variable names, are normalized to lower case at the parsing stage. The
content of string literals is left untouched.

Keywords are all letters; variable and function names begin with a letter,
followed by any number of letters and digits, and optionally end with a dollar
sign ("$") to denote a string type. Numbers are made of digits, and may have
a decimal point in the middle; scientific notation isn't yet supported, and
an initial minus sign is parsed as the negation operator due to a quirk.

Literal character strings are delimited by double quote signs; there is no
way to directly embed the delimiter in a string at this point.
"""

help_text["variables"] = """
Batch Basic supports two types of variables: strings and numbers (64-bit
floating point). End a variable name with a dollar sign ("$") to make it
of type string. Function names and arguments behave in the same way, except
argument types are checked at runtime, as opposed to parse time like with
other names. A and A$ are different variables that can exist at the same
time. A function and a variable can have the same name.

A few variable names have special meaning in Batch Basic:

PROMPT$	will have its content used as the command prompt in interactive mode,
	if it's defined.
ERR	will be set to the exit code of the last external program invoked
	via the SHELL operator or the "!" command.
ERR$	will hold any error message resulting from the same, or else the
	empty string.
"""

help_text["special"] = """
Several Batch Basic operators look more like functions, except they can only
appear at the start of an expression:

IIF(condition, then, else)
	evaluates the condition; depending on whether it's true or false, it
	then evaluates and returns the "then" or "else" branch; these must be
	of the same type.
MIN(expression, [, expression...])
	returns the smallest of its arguments; all of them must be of the same
	type, either numbers or strings.
MAX(expression, [, expression...])
	returns the largest of its arguments; all of them must be of the same
	type, either numbers or strings.
SHELL(program [, argument...])
	runs the given program, passing it any given arguments on the command
	line, safely quoted, and returns any output from the program; two
	special variables, ERR and ERR$, will be set to the program's exit
	code and any error message, respectively.
PATH(string, [, string...])
	returns its arguments joined by the correct path separator for the
	host operating system.
"""

version = "2017-03-06 beta"

banner = """
Batch Basic: expression calculator and OS shell, version {0}.
Type BYE to quit or HELP for usage instructions.
"""

if __name__ == "__main__":
	try:
		import readline
	except ImportError as e:
		print("(Command line editing is unavailable.)\n")

	import argparse

	pargs = argparse.ArgumentParser(
		description='Batch Basic: expression calculator / OS shell')
	pargs.add_argument("-V", "--version", action="store_true",
		help="show version number and exit")
	pargs.add_argument("-b", "--batch", action="store_true",
		help="exit after loading the files given as arguments")
	pargs.add_argument("-r", "--run", dest="code",
		help="run given code instead of going interactive")
	pargs.add_argument("preload", type=argparse.FileType('r'), nargs="*",
		help="load given file before going interactive")
	args = pargs.parse_args()
	
	if args.version:
		print("Batch Basic version " + version)
	else:
		scope = Scope()
		functions = {}
		for handle in args.preload:
			load_file(handle, scope, functions)
			handle.close()
		if args.code:
			parser = Parser(args.code)
			try:
				eval_line(parse_line(parser), scope, functions)
			except Exception as e:
				print("Column {0}: {1}.".format(parser.pos, e))
		elif not args.batch:
			repl(banner.format(version), scope, functions)
