## BASIC-like expression calculator and operating system shell.

import strutils, nre, options

const statement_names = [
  "print", "let", "def", "load", "cd",
  "if", "while", "data", "read", "restore",
  "mkdir", "rmdir", "copy", "name", "delete"]

# Each RE must have exactly one parenthesized group, even if not needed.
let re_stmt = re(r"(?i)\s*(" & join(statement_names, "|") & r")\b")
let re_fn = re r"(?i)\s*(fn)" # Allow for DEF FNa$ = ...
let re_then = re r"(?i)\s*(then)\b"
let re_else = re r"(?i)\s*(else)\b"
let re_wend = re r"(?i)\s*(wend)\b"
let re_name = re r"(?i)\s*([a-z][a-z0-9]*\$?)"
let re_float* = re r"\s*(-?\d+(\.\d*)?)"
let re_string* = re r"\s*""([^""]*)"""
let re_rem = re r"\s*'(.*)$"
let re_eol = re r"(\s*)$"
let re_assign = re r"\s*(=)"
let re_specop = re r"(?i)\s*(iif|min|max|shell|path)\b"
let re_or = re r"(?i)\s*(or)\b"
let re_and = re r"(?i)\s*(and)\b"
let re_not = re r"(?i)\s*(not)\b"
let re_cmp = re r"\s*(<>|=|<=|<|>=|>)"
let re_addop = re r"\s*([+-])"
let re_mulop = re r"\s*([*/\\])"
let re_powop = re r"\s*(^)"
let re_lparen = re r"\s*(\()"
let re_rparen = re r"\s*(\))"
let re_comma = re r"\s*(,)"
let re_semi = re r"\s*(;)"
let re_colon = re r"\s*(:)"


type Parser* = ref object of RootObj
  text: string
  pos*: int
  token: string

proc match(self: Parser, regex: Regex): bool =
  let optm = match(self.text, regex, self.pos)
  if optm.isNone:
    return false
  else:
    let m = optm.get()
    self.token = m.captures[0]
    self.pos = m.matchBounds.b
    return true

proc peek(self: Parser, regex: Regex): bool =
  return not match(self.text, regex, self.pos).isNone


type BasType = enum
  numType, strType

type ExprKind = enum
  exLitNum, exLitStr, exVar#, exFunCall,
  #exBinaryOp, exUnaryOp, exSpecialOp
  
type Expression = object of RootObj
  extype: BasType
  case kind: ExprKind
  of exLitNum:
    numval: float
  of exLitStr:
    strval: string
  of exVar:
    varname: string

type StmtKind = enum
  stPrint, stLet#, stDef, stLoad, stCd,
  #stIf, stWhile, stData, stRead, stRestore,
  #stMkdir, stRmdir, stCopy, stName, stDelete

type Statement = object of RootObj
  case kind: StmtKind
  of stPrint:
    exprlist: seq[Expression]
  of stLet:
    varname: Expression
    letexpr: Expression

type SyntaxError* = object of Exception
type TypeError* = object of Exception


proc parse_stmtlist(parser: Parser): seq[Statement]

proc parse_line*(parser: Parser): seq[Statement] =
  let stlist = parse_stmtlist(parser)

  if parser.match(re_rem) or parser.match(re_eol):
    return stlist
  else:
    raise newException(SyntaxError, "statement or comment expected")

proc parse_statement(parser: Parser): Statement
proc parse_let(parser: Parser): Statement

proc parse_stmtlist(parser: Parser): seq[Statement] =
  var stlist: seq[Statement] = @[]
  var go_on = false

  if parser.match(re_stmt):
    stlist.add parse_statement(parser)
    go_on = true
  elif parser.peek(re_name):
    stlist.add parse_let(parser)
    go_on = true
  
  if go_on:
    while parser.match(re_colon):
      if parser.match(re_stmt):
        stlist.add parse_statement(parser)
      elif parser.peek(re_name):
        stlist.add parse_let(parser)
      else:
        break
  return stlist

proc parse_print(parser: Parser): Statement

proc parse_statement(parser: Parser): Statement =
  let stm = parser.token.toLowerAscii
  if stm == "print":
    return parse_print(parser)
  elif stm == "let":
    return parse_let(parser)
  else:
    raise newException(SyntaxError, "not a statement: " & parser.token)

proc at_stmt_end(parser: Parser): bool =
  parser.peek(re_colon) or parser.peek(re_rem) or parser.peek(re_eol)

proc parse_expression(parser: Parser): Expression

proc parse_print(parser: Parser): Statement =
  var stm = Statement(kind: stPrint, exprlist: @[])
  
  if at_stmt_end(parser):
    return stm
  
  stm.exprlist.add parse_expression(parser)
  
  while parser.match(re_comma) or parser.match(re_semi):
    if at_stmt_end(parser):
      return stm
    if parser.token == ",":
      stm.exprlist.add Expression(
        extype: strType, kind: exLitStr, strval: "\t\t")
    stm.exprlist.add parse_expression(parser)

  stm.exprlist.add Expression(
    extype: strType, kind: exLitStr, strval: "\n")
  return stm

proc name2ref(name: string): Expression =
  if name[-1] == '$':
    return Expression(extype: strType, kind: exVar, varname: name)
  else:
    return Expression(extype: numType, kind: exVar, varname: name)

proc parse_let(parser: Parser): Statement =
  if not parser.match(re_name):
    raise newException(SyntaxError, "variable name expected")
  
  let lside = name2ref parser.token.toLowerAscii
  
  if not parser.match(re_assign):
    raise newException(SyntaxError, "assignment expected")
  
  let rside = parse_expression parser
  
  if lside.extype == rside.extype:
    return Statement(kind: stLet, varname: lside, letexpr: rside)
  else:
    raise newException(TypeError, "type mismatch in assignment")


proc parse_expression(parser: Parser): Expression =
  Expression(extype: numType, kind: exLitNum, numval: 10)


const version = "2017-03-06 beta"
const banner = """
Batch Basic: expression calculator and OS shell, version $1.
Type BYE to quit or HELP for usage instructions.
"""

when isMainModule:
  echo banner % version
